/*
 * Copyright (C) 2019,2020 by Sukchan Lee <acetcom@gmail.com>
 *
 * This file is part of Open5GS.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include "ngap-handler.h"
#include "ngap-path.h"
#include "sbi-path.h"
#include "nas-path.h"

static bool served_tai_is_found(amf_gnb_t *gnb)
{
    int i, j;
    int served_tai_index;

    for (i = 0; i < gnb->num_of_supported_ta_list; i++) {
        for (j = 0; j < gnb->supported_ta_list[i].num_of_bplmn_list; j++) {
            ogs_5gs_tai_t tai;
            memcpy(&tai.plmn_id,
                    &gnb->supported_ta_list[i].bplmn_list[j].plmn_id,
                        OGS_PLMN_ID_LEN);
            tai.tac.v = gnb->supported_ta_list[i].tac.v;
            served_tai_index = amf_find_served_tai(&tai);
            if (served_tai_index >= 0 &&
                    served_tai_index < OGS_MAX_NUM_OF_SERVED_TAI) {
                ogs_debug("    TAC[%d]", gnb->supported_ta_list[i].tac.v);
                ogs_debug("    PLMN_ID[MCC:%d MNC:%d]",
                    ogs_plmn_id_mcc(&gnb->supported_ta_list[i].
                        bplmn_list[j].plmn_id),
                    ogs_plmn_id_mnc(&gnb->supported_ta_list[i].
                        bplmn_list[j].plmn_id));
                ogs_debug("    SERVED_TAI_INDEX[%d]", served_tai_index);
                return true;
            }
        }
    }

    return false;
}

static bool s_nssai_is_found(amf_gnb_t *gnb)
{
    int i, j, k;
    ogs_s_nssai_t *s_nssai = NULL;

    for (i = 0; i < gnb->num_of_supported_ta_list; i++) {
        for (j = 0; j < gnb->supported_ta_list[i].num_of_bplmn_list; j++) {
            for (k = 0; k < gnb->supported_ta_list[i].
                                bplmn_list[j].num_of_s_nssai; k++) {
                s_nssai = amf_find_s_nssai(
                    &gnb->supported_ta_list[i].bplmn_list[j].plmn_id,
                    &gnb->supported_ta_list[i].bplmn_list[j].s_nssai[k]);
                if (s_nssai) {
                    ogs_debug("    PLMN_ID[MCC:%d MNC:%d]",
                        ogs_plmn_id_mcc(&gnb->supported_ta_list[i].
                            bplmn_list[j].plmn_id),
                        ogs_plmn_id_mnc(&gnb->supported_ta_list[i].
                            bplmn_list[j].plmn_id));
                    ogs_debug("    S_NSSAI[SST:%d SD:0x%x]",
                        gnb->supported_ta_list[i].bplmn_list[j].s_nssai[k].sst,
                        gnb->supported_ta_list[i].bplmn_list[j].
                                    s_nssai[k].sd.v);
                    return true;
                }
            }
        }
    }

    return false;
}

static bool maximum_number_of_gnbs_is_reached(void)
{
    amf_gnb_t *gnb = NULL, *next_gnb = NULL;
    int number_of_gnbs_online = 0;

    ogs_list_for_each_safe(&amf_self()->gnb_list, next_gnb, gnb) {
        if (gnb->state.ng_setup_success) {
            number_of_gnbs_online++;
        }
    }

    return number_of_gnbs_online >= ogs_app()->max.gnb;
}

void ngap_handle_ng_setup_request(amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i, j, k;

    NGAP_InitiatingMessage_t *initiatingMessage = NULL;
    NGAP_NGSetupRequest_t *NGSetupRequest = NULL;

    NGAP_NGSetupRequestIEs_t *ie = NULL;
    NGAP_GlobalRANNodeID_t *GlobalRANNodeID = NULL;
    NGAP_GlobalGNB_ID_t *globalGNB_ID = NULL;
    NGAP_SupportedTAList_t *SupportedTAList = NULL;
    NGAP_PagingDRX_t *PagingDRX = NULL;

    NGAP_Cause_PR group = NGAP_Cause_PR_NOTHING;
    long cause = 0;

    uint32_t gnb_id;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    initiatingMessage = message->choice.initiatingMessage;
    ogs_assert(initiatingMessage);
    NGSetupRequest = &initiatingMessage->value.choice.NGSetupRequest;
    ogs_assert(NGSetupRequest);

    ogs_debug("NGSetupRequest");

    for (i = 0; i < NGSetupRequest->protocolIEs.list.count; i++) {
        ie = NGSetupRequest->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_GlobalRANNodeID:
            GlobalRANNodeID = &ie->value.choice.GlobalRANNodeID;
            break;
        case NGAP_ProtocolIE_ID_id_SupportedTAList:
            SupportedTAList = &ie->value.choice.SupportedTAList;
            break;
        case NGAP_ProtocolIE_ID_id_DefaultPagingDRX:
            PagingDRX = &ie->value.choice.PagingDRX;
            break;
        default:
            break;
        }
    }

    if (!GlobalRANNodeID) {
        ogs_error("No GlobalRANNodeID");
        group = NGAP_Cause_PR_protocol;
        cause = NGAP_CauseProtocol_semantic_error;
        ngap_send_ng_setup_failure(gnb, group, cause);
        return;
    }

    globalGNB_ID = GlobalRANNodeID->choice.globalGNB_ID;
    if (!globalGNB_ID) {
        ogs_error("No globalGNB_ID");
        group = NGAP_Cause_PR_protocol;
        cause = NGAP_CauseProtocol_semantic_error;
        ngap_send_ng_setup_failure(gnb, group, cause);
        return;
    }

    if (!SupportedTAList) {
        ogs_error("No SupportedTAList");
        group = NGAP_Cause_PR_protocol;
        cause = NGAP_CauseProtocol_semantic_error;
        ngap_send_ng_setup_failure(gnb, group, cause);
        return;
    }

    ogs_ngap_GNB_ID_to_uint32(&globalGNB_ID->gNB_ID, &gnb_id);
    ogs_debug("    IP[%s] GNB_ID[0x%x]", OGS_ADDR(gnb->sctp.addr, buf), gnb_id);

    if (PagingDRX)
        ogs_debug("    PagingDRX[%ld]", *PagingDRX);

    /* Parse Supported TA */
    for (i = 0, gnb->num_of_supported_ta_list = 0;
            i < SupportedTAList->list.count &&
            gnb->num_of_supported_ta_list < OGS_MAX_NUM_OF_TAI;
                i++) {
        NGAP_SupportedTAItem_t *SupportedTAItem = NULL;

        SupportedTAItem = (NGAP_SupportedTAItem_t *)
                SupportedTAList->list.array[i];
        if (!SupportedTAItem) {
            ogs_error("No SupportedTAItem");
            group = NGAP_Cause_PR_protocol;
            cause = NGAP_CauseProtocol_semantic_error;
            ngap_send_ng_setup_failure(gnb, group, cause);
            return;
        }

        ogs_asn_OCTET_STRING_to_uint24(&SupportedTAItem->tAC,
            &gnb->supported_ta_list[i].tac);

        ogs_debug("    TAC[%d]", gnb->supported_ta_list[i].tac.v);

        for (j = 0, gnb->supported_ta_list[i].num_of_bplmn_list = 0;
                j < SupportedTAItem->broadcastPLMNList.list.count &&
                gnb->supported_ta_list[i].num_of_bplmn_list <
                    OGS_MAX_NUM_OF_BPLMN;
                        j++) {

            NGAP_BroadcastPLMNItem_t *BroadcastPLMNItem = NULL;
            NGAP_PLMNIdentity_t *pLMNIdentity = NULL;

            BroadcastPLMNItem = (NGAP_BroadcastPLMNItem_t *)
                    SupportedTAItem->broadcastPLMNList.list.array[j];
            if (!BroadcastPLMNItem) {
                ogs_error("No BroadcastPLMNItem");
                group = NGAP_Cause_PR_protocol;
                cause = NGAP_CauseProtocol_semantic_error;
                ngap_send_ng_setup_failure(gnb, group, cause);
                return;
            }

            pLMNIdentity = (NGAP_PLMNIdentity_t *)
                    &BroadcastPLMNItem->pLMNIdentity;
            ogs_assert(pLMNIdentity);

            memcpy(&gnb->supported_ta_list[i].bplmn_list[j].plmn_id,
                    pLMNIdentity->buf, sizeof(ogs_plmn_id_t));
            ogs_debug("    PLMN_ID[MCC:%d MNC:%d]",
                ogs_plmn_id_mcc(&gnb->supported_ta_list[i].
                    bplmn_list[j].plmn_id),
                ogs_plmn_id_mnc(&gnb->supported_ta_list[i].
                    bplmn_list[j].plmn_id));

            for (k = 0, gnb->supported_ta_list[i].
                            bplmn_list[j].num_of_s_nssai = 0;
                    k < BroadcastPLMNItem->tAISliceSupportList.list.count &&
                    gnb->supported_ta_list[i].bplmn_list[j].num_of_s_nssai <
                        OGS_MAX_NUM_OF_SLICE;
                            k++) {
                NGAP_SliceSupportItem_t *SliceSupportItem = NULL;
                NGAP_S_NSSAI_t *s_NSSAI = NULL;

                SliceSupportItem = (NGAP_SliceSupportItem_t *)
                        BroadcastPLMNItem->tAISliceSupportList.list.array[k];
                if (!SliceSupportItem) {
                    ogs_error("No SliceSupportItem");
                    group = NGAP_Cause_PR_protocol;
                    cause = NGAP_CauseProtocol_semantic_error;
                    ngap_send_ng_setup_failure(gnb, group, cause);
                    return;
                }

                s_NSSAI = &SliceSupportItem->s_NSSAI;
                ogs_assert(s_NSSAI);

                ogs_asn_OCTET_STRING_to_uint8(&s_NSSAI->sST,
                    &gnb->supported_ta_list[i].bplmn_list[j].s_nssai[k].sst);
                if (!s_NSSAI->sD) {
                    gnb->supported_ta_list[i].bplmn_list[j].s_nssai[k].sd.v =
                        OGS_S_NSSAI_NO_SD_VALUE;
                } else {
                    ogs_asn_OCTET_STRING_to_uint24(s_NSSAI->sD,
                    &gnb->supported_ta_list[i].bplmn_list[j].s_nssai[k].sd);
                }

                ogs_debug("    S_NSSAI[SST:%d SD:0x%x]",
                    gnb->supported_ta_list[i].bplmn_list[j].s_nssai[k].sst,
                    gnb->supported_ta_list[i].bplmn_list[j].s_nssai[k].sd.v);

                gnb->supported_ta_list[i].bplmn_list[j].num_of_s_nssai++;
            }

            gnb->supported_ta_list[i].num_of_bplmn_list++;
        }

        gnb->num_of_supported_ta_list++;
    }

    if (maximum_number_of_gnbs_is_reached()) {
        ogs_warn("NG-Setup failure:");
        ogs_warn("    Maximum number of gNBs reached");
        group = NGAP_Cause_PR_misc;
        cause = NGAP_CauseMisc_control_processing_overload;

        ngap_send_ng_setup_failure(gnb, group, cause);
        return;
    }

    if (gnb->num_of_supported_ta_list == 0) {
        ogs_warn("NG-Setup failure:");
        ogs_warn("    No supported TA exist in NG-Setup request");
        group = NGAP_Cause_PR_protocol;
        cause = NGAP_CauseProtocol_message_not_compatible_with_receiver_state;

        ngap_send_ng_setup_failure(gnb, group, cause);
        return;
    }

    if (!served_tai_is_found(gnb)) {
        ogs_warn("NG-Setup failure:");
        ogs_warn("    Cannot find Served TAI. Check 'amf.tai' configuration");
        group = NGAP_Cause_PR_misc;
        cause = NGAP_CauseMisc_unknown_PLMN;

        ngap_send_ng_setup_failure(gnb, group, cause);
        return;
    }

    if (!s_nssai_is_found(gnb)) {
        ogs_warn("NG-Setup failure:");
        ogs_warn("    Cannot find S_NSSAI. "
                    "Check 'amf.plmn_support.s_nssai' configuration");
        group = NGAP_Cause_PR_misc;
        cause = NGAP_CauseMisc_unknown_PLMN;

        ngap_send_ng_setup_failure(gnb, group, cause);
        return;
    }

    amf_gnb_set_gnb_id(gnb, gnb_id);

    gnb->state.ng_setup_success = true;
    ngap_send_ng_setup_response(gnb);
}

void ngap_handle_initial_ue_message(amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    int i;
    char buf[OGS_ADDRSTRLEN];

    ran_ue_t *ran_ue = NULL;

    NGAP_InitiatingMessage_t *initiatingMessage = NULL;
    NGAP_InitialUEMessage_t *InitialUEMessage = NULL;

    NGAP_InitialUEMessage_IEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_NAS_PDU_t *NAS_PDU = NULL;
    NGAP_UserLocationInformation_t *UserLocationInformation = NULL;
    NGAP_UserLocationInformationNR_t *UserLocationInformationNR = NULL;
    NGAP_FiveG_S_TMSI_t *FiveG_S_TMSI = NULL;
    NGAP_UEContextRequest_t *UEContextRequest = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    initiatingMessage = message->choice.initiatingMessage;
    ogs_assert(initiatingMessage);
    InitialUEMessage = &initiatingMessage->value.choice.InitialUEMessage;
    ogs_assert(InitialUEMessage);

    ogs_info("InitialUEMessage");

    for (i = 0; i < InitialUEMessage->protocolIEs.list.count; i++) {
        ie = InitialUEMessage->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_NAS_PDU:
            NAS_PDU = &ie->value.choice.NAS_PDU;
            break;
        case NGAP_ProtocolIE_ID_id_UserLocationInformation:
            UserLocationInformation =
                &ie->value.choice.UserLocationInformation;
            break;
        case NGAP_ProtocolIE_ID_id_FiveG_S_TMSI:
            FiveG_S_TMSI = &ie->value.choice.FiveG_S_TMSI;
            break;
        case NGAP_ProtocolIE_ID_id_UEContextRequest:
            UEContextRequest = &ie->value.choice.UEContextRequest;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!RAN_UE_NGAP_ID) {
        ogs_error("No RAN_UE_NGAP_ID");
        ngap_send_error_indication(gnb, NULL, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    ran_ue = ran_ue_find_by_ran_ue_ngap_id(gnb, *RAN_UE_NGAP_ID);
    if (!ran_ue) {
        ran_ue = ran_ue_add(gnb, *RAN_UE_NGAP_ID);
        ogs_assert(ran_ue);

        /* Find AMF_UE if 5G-S_TMSI included */
        if (FiveG_S_TMSI) {
            ogs_nas_5gs_guti_t nas_guti;
            amf_ue_t *amf_ue = NULL;
            uint8_t region;
            uint16_t set;
            uint8_t pointer;
            uint32_t m_tmsi;

            memset(&nas_guti, 0, sizeof(ogs_nas_5gs_guti_t));

            /* Use the first configured plmn_id and mme group id */
            ogs_nas_from_plmn_id(&nas_guti.nas_plmn_id,
                    &amf_self()->served_guami[0].plmn_id);
            region = amf_self()->served_guami[0].amf_id.region;

            /* Getting from 5G-S_TMSI */
            ogs_ngap_AMFSetID_to_uint16(&FiveG_S_TMSI->aMFSetID, &set);
            ogs_ngap_AMFPointer_to_uint8(&FiveG_S_TMSI->aMFPointer, &pointer);

            ogs_amf_id_build(&nas_guti.amf_id, region, set, pointer);

            /* size must be 4 */
            ogs_asn_OCTET_STRING_to_uint32(&FiveG_S_TMSI->fiveG_TMSI, &m_tmsi);
            nas_guti.m_tmsi = m_tmsi;

            amf_ue = amf_ue_find_by_guti(&nas_guti);
            if (!amf_ue) {
                ogs_info("Unknown UE by 5G-S_TMSI[AMF_ID:0x%x,M_TMSI:0x%x]",
                    ogs_amf_id_hexdump(&nas_guti.amf_id), nas_guti.m_tmsi);
            } else {
                ogs_info("[%s]    5G-S_TMSI[AMF_ID:0x%x,M_TMSI:0x%x]",
                        AMF_UE_HAVE_SUCI(amf_ue) ? amf_ue->suci : "Unknown ID",
                        ogs_amf_id_hexdump(&amf_ue->current.guti.amf_id),
                        amf_ue->current.guti.m_tmsi);
                /* If NAS(amf_ue_t) has already been associated with
                 * older NG(ran_ue_t) context */
                if (CM_CONNECTED(amf_ue)) {
                    /* Previous NG(ran_ue_t) context the holding timer(30secs)
                     * is started.
                     * Newly associated NG(ran_ue_t) context holding timer
                     * is stopped. */
                    ogs_debug("[%s] Start NG Holding Timer", amf_ue->suci);
                    ogs_debug("[%s]    RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld]",
                            amf_ue->suci, amf_ue->ran_ue->ran_ue_ngap_id,
                            (long long)amf_ue->ran_ue->amf_ue_ngap_id);

                    /* De-associate NG with NAS/EMM */
                    ran_ue_deassociate(amf_ue->ran_ue);

                    ngap_send_ran_ue_context_release_command(amf_ue->ran_ue,
                            NGAP_Cause_PR_nas, NGAP_CauseNas_normal_release,
                            NGAP_UE_CTX_REL_NG_CONTEXT_REMOVE, 0);
                }
                amf_ue_associate_ran_ue(amf_ue, ran_ue);
            }
        }
    }

    if (!UserLocationInformation) {
        ogs_error("No UserLocationInformation");
        ngap_send_error_indication(gnb, &ran_ue->ran_ue_ngap_id, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (UserLocationInformation->present !=
            NGAP_UserLocationInformation_PR_userLocationInformationNR) {
        ogs_error("Not implemented UserLocationInformation[%d]",
                UserLocationInformation->present);
        ngap_send_error_indication(gnb, &ran_ue->ran_ue_ngap_id, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_unspecified);
        return;
    }

    if (!NAS_PDU) {
        ogs_error("No NAS_PDU");
        ngap_send_error_indication(gnb, &ran_ue->ran_ue_ngap_id, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    UserLocationInformationNR =
        UserLocationInformation->choice.userLocationInformationNR;
    ogs_assert(UserLocationInformationNR);
    ogs_ngap_ASN_to_nr_cgi(
            &UserLocationInformationNR->nR_CGI, &ran_ue->saved.nr_cgi);
    ogs_ngap_ASN_to_5gs_tai(
            &UserLocationInformationNR->tAI, &ran_ue->saved.nr_tai);

    ogs_info("    RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] "
            "TAC[%d] CellID[0x%llx]",
        ran_ue->ran_ue_ngap_id, (long long)ran_ue->amf_ue_ngap_id,
        ran_ue->saved.nr_tai.tac.v, (long long)ran_ue->saved.nr_cgi.cell_id);

    if (UEContextRequest) {
        if (*UEContextRequest == NGAP_UEContextRequest_requested) {
            ran_ue->ue_context_requested = true;
        }
    }

    ngap_send_to_nas(ran_ue, NGAP_ProcedureCode_id_InitialUEMessage, NAS_PDU);
}

void ngap_handle_uplink_nas_transport(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    amf_ue_t *amf_ue = NULL;
    ran_ue_t *ran_ue = NULL;
    uint64_t amf_ue_ngap_id;

    NGAP_InitiatingMessage_t *initiatingMessage = NULL;
    NGAP_UplinkNASTransport_t *UplinkNASTransport = NULL;

    NGAP_UplinkNASTransport_IEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_NAS_PDU_t *NAS_PDU = NULL;
    NGAP_UserLocationInformation_t *UserLocationInformation = NULL;
    NGAP_UserLocationInformationNR_t *UserLocationInformationNR = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    initiatingMessage = message->choice.initiatingMessage;
    ogs_assert(initiatingMessage);
    UplinkNASTransport = &initiatingMessage->value.choice.UplinkNASTransport;
    ogs_assert(UplinkNASTransport);

    ogs_debug("UplinkNASTransport");

    for (i = 0; i < UplinkNASTransport->protocolIEs.list.count; i++) {
        ie = UplinkNASTransport->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_NAS_PDU:
            NAS_PDU = &ie->value.choice.NAS_PDU;
            break;
        case NGAP_ProtocolIE_ID_id_UserLocationInformation:
            UserLocationInformation = &ie->value.choice.UserLocationInformation;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    ran_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!ran_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    amf_ue = ran_ue->amf_ue;
    if (!amf_ue) {
        ogs_error("Cannot find AMF-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &ran_ue->ran_ue_ngap_id, &ran_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    if (!UserLocationInformation) {
        ogs_error("No UserLocationInformation");
        ngap_send_error_indication(gnb, &ran_ue->ran_ue_ngap_id, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (UserLocationInformation->present !=
            NGAP_UserLocationInformation_PR_userLocationInformationNR) {
        ogs_error("Not implemented UserLocationInformation[%d]",
                UserLocationInformation->present);
        ngap_send_error_indication(gnb, &ran_ue->ran_ue_ngap_id, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_unspecified);
        return;
    }

    if (!NAS_PDU) {
        ogs_error("No NAS_PDU");
        ngap_send_error_indication(
            gnb, &ran_ue->ran_ue_ngap_id, &ran_ue->amf_ue_ngap_id,
            NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    UserLocationInformationNR =
        UserLocationInformation->choice.userLocationInformationNR;
    ogs_assert(UserLocationInformationNR);
    ogs_ngap_ASN_to_nr_cgi(
            &UserLocationInformationNR->nR_CGI, &ran_ue->saved.nr_cgi);
    ogs_ngap_ASN_to_5gs_tai(
            &UserLocationInformationNR->tAI, &ran_ue->saved.nr_tai);

    ogs_debug("    RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] "
            "TAC[%d] CellID[0x%llx]",
        ran_ue->ran_ue_ngap_id, (long long)ran_ue->amf_ue_ngap_id,
        ran_ue->saved.nr_tai.tac.v, (long long)ran_ue->saved.nr_cgi.cell_id);

    /* Copy NR-TAI/NR-CGI from ran_ue */
    memcpy(&amf_ue->nr_tai, &ran_ue->saved.nr_tai, sizeof(ogs_5gs_tai_t));
    memcpy(&amf_ue->nr_cgi, &ran_ue->saved.nr_cgi, sizeof(ogs_nr_cgi_t));

    ngap_send_to_nas(ran_ue, NGAP_ProcedureCode_id_UplinkNASTransport, NAS_PDU);
}

void ngap_handle_ue_radio_capability_info_indication(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    ran_ue_t *ran_ue = NULL;
    uint64_t amf_ue_ngap_id;

    NGAP_InitiatingMessage_t *initiatingMessage = NULL;
    NGAP_UERadioCapabilityInfoIndication_t
        *UERadioCapabilityInfoIndication = NULL;

    NGAP_UERadioCapabilityInfoIndicationIEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_UERadioCapability_t *UERadioCapability = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    initiatingMessage = message->choice.initiatingMessage;
    ogs_assert(initiatingMessage);
    UERadioCapabilityInfoIndication =
        &initiatingMessage->value.choice.UERadioCapabilityInfoIndication;
    ogs_assert(UERadioCapabilityInfoIndication);

    ogs_debug("UERadioCapabilityInfoIndication");

    for (i = 0;
            i < UERadioCapabilityInfoIndication->protocolIEs.list.count; i++) {
        ie = UERadioCapabilityInfoIndication->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_UERadioCapability:
            UERadioCapability = &ie->value.choice.UERadioCapability;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    ran_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!ran_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    ogs_debug("    RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld]",
            ran_ue->ran_ue_ngap_id, (long long)ran_ue->amf_ue_ngap_id);

    if (!UERadioCapability) {
        ogs_error("No UERadioCapability");
        ngap_send_error_indication(
                gnb, &ran_ue->ran_ue_ngap_id, &ran_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (ran_ue->amf_ue)
        OGS_ASN_STORE_DATA(&ran_ue->amf_ue->ueRadioCapability,
                UERadioCapability);
}

void ngap_handle_initial_context_setup_response(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    amf_ue_t *amf_ue = NULL;
    ran_ue_t *ran_ue = NULL;
    amf_sess_t *sess = NULL;
    uint64_t amf_ue_ngap_id;
    amf_nsmf_pdusession_update_sm_context_param_t param;

    bool paging_ongoing = false;

    NGAP_SuccessfulOutcome_t *successfulOutcome = NULL;
    NGAP_InitialContextSetupResponse_t *InitialContextSetupResponse = NULL;

    NGAP_InitialContextSetupResponseIEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_PDUSessionResourceSetupListCxtRes_t *PDUSessionList = NULL;
    NGAP_PDUSessionResourceSetupItemCxtRes_t *PDUSessionItem = NULL;
    OCTET_STRING_t *transfer = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    successfulOutcome = message->choice.successfulOutcome;
    ogs_assert(successfulOutcome);
    InitialContextSetupResponse =
        &successfulOutcome->value.choice.InitialContextSetupResponse;
    ogs_assert(InitialContextSetupResponse);

    ogs_debug("InitialContextSetupResponse");

    for (i = 0; i < InitialContextSetupResponse->protocolIEs.list.count; i++) {
        ie = InitialContextSetupResponse->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_PDUSessionResourceSetupListCxtRes:
            PDUSessionList =
                &ie->value.choice.PDUSessionResourceSetupListCxtRes;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    ran_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!ran_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    ogs_debug("    RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld]",
            ran_ue->ran_ue_ngap_id, (long long)ran_ue->amf_ue_ngap_id);

    amf_ue = ran_ue->amf_ue;
    if (!amf_ue) {
        ogs_error("Cannot find AMF-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &ran_ue->ran_ue_ngap_id, &ran_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    for (i = 0; PDUSessionList && i < PDUSessionList->list.count; i++) {
        PDUSessionItem = (NGAP_PDUSessionResourceSetupItemCxtRes_t *)
            PDUSessionList->list.array[i];

        if (!PDUSessionItem) {
            ogs_error("No PDUSessionResourceSetupItemCxtRes");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        transfer = &PDUSessionItem->pDUSessionResourceSetupResponseTransfer;
        if (!transfer) {
            ogs_error("No PDUSessionResourceSetupResponseTransfer");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        if (PDUSessionItem->pDUSessionID ==
                OGS_NAS_PDU_SESSION_IDENTITY_UNASSIGNED) {
            ogs_error("PDU Session Identity is unassigned");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        sess = amf_sess_find_by_psi(amf_ue, PDUSessionItem->pDUSessionID);
        if (!sess) {
            ogs_error("Cannot find PDU Session ID [%d]",
                    (int)PDUSessionItem->pDUSessionID);
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_radioNetwork,
                    NGAP_CauseRadioNetwork_unknown_PDU_session_ID);
            return;
        }

        if (!SESSION_CONTEXT_IN_SMF(sess)) {
            ogs_error("Session Context is not in SMF [%d]",
                    (int)PDUSessionItem->pDUSessionID);
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_radioNetwork,
                    NGAP_CauseRadioNetwork_unknown_PDU_session_ID);
            return;
        }

        memset(&param, 0, sizeof(param));
        param.n2smbuf = ogs_pkbuf_alloc(NULL, OGS_MAX_SDU_LEN);
        ogs_assert(param.n2smbuf);
        param.n2SmInfoType = OpenAPI_n2_sm_info_type_PDU_RES_SETUP_RSP;
        ogs_pkbuf_put_data(param.n2smbuf, transfer->buf, transfer->size);

        amf_sess_sbi_discover_and_send(OpenAPI_nf_type_SMF,
                sess, AMF_UPDATE_SM_CONTEXT_ACTIVATED, &param,
                amf_nsmf_pdusession_build_update_sm_context);

        ogs_pkbuf_free(param.n2smbuf);
    }

    ogs_list_for_each(&amf_ue->sess_list, sess) {
        if (sess->paging.ongoing == true) {
            paging_ongoing = true;
            break;
        }
    }

    if (paging_ongoing == true) {
        gmm_configuration_update_command_param_t param;

        /* Create New GUTI */
        amf_ue_new_guti(amf_ue);

        memset(&param, 0, sizeof(param));
        param.acknowledgement_requested = 1;
        param.guti = 1;
        nas_5gs_send_configuration_update_command(amf_ue, &param);
    }

    AMF_UE_CLEAR_PAGING_INFO(amf_ue);
}

void ngap_handle_initial_context_setup_failure(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i, old_xact_count = 0, new_xact_count = 0;

    ran_ue_t *ran_ue = NULL;
    amf_ue_t *amf_ue = NULL;
    uint64_t amf_ue_ngap_id;

    NGAP_UnsuccessfulOutcome_t *unsuccessfulOutcome = NULL;
    NGAP_InitialContextSetupFailure_t *InitialContextSetupFailure = NULL;

    NGAP_InitialContextSetupFailureIEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_Cause_t *Cause = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    unsuccessfulOutcome = message->choice.unsuccessfulOutcome;
    ogs_assert(unsuccessfulOutcome);
    InitialContextSetupFailure =
        &unsuccessfulOutcome->value.choice.InitialContextSetupFailure;
    ogs_assert(InitialContextSetupFailure);

    ogs_debug("InitialContextSetupFailure");

    for (i = 0; i < InitialContextSetupFailure->protocolIEs.list.count; i++) {
        ie = InitialContextSetupFailure->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_Cause:
            Cause = &ie->value.choice.Cause;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    ran_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!ran_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    ogs_debug("    RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld]",
            ran_ue->ran_ue_ngap_id, (long long)ran_ue->amf_ue_ngap_id);

    if (!Cause) {
        ogs_error("No Cause");
        ngap_send_error_indication(
                gnb, &ran_ue->ran_ue_ngap_id, &ran_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }
    ogs_debug("    Cause[Group:%d Cause:%d]",
            Cause->present, (int)Cause->choice.radioNetwork);

    /*
     * 19.2.2.3 in Spec 36.300
     *
     * In case of failure, RAN and AMF behaviours are not mandated.
     *
     * Both implicit release (local release at each node) and
     * explicit release (AMF-initiated UE Context Release procedure)
     * may in principle be adopted. The RAN should ensure
     * that no hanging resources remain at the RAN.
     */
    amf_ue = ran_ue->amf_ue;
    if (amf_ue) {
        old_xact_count = amf_sess_xact_count(amf_ue);

        amf_ue->deactivation.group = NGAP_Cause_PR_nas;
        amf_ue->deactivation.cause = NGAP_CauseNas_normal_release;

        amf_sbi_send_deactivate_all_sessions(
                amf_ue, AMF_UPDATE_SM_CONTEXT_DEACTIVATED,
                Cause->present, (int)Cause->choice.radioNetwork);

        new_xact_count = amf_sess_xact_count(amf_ue);
    }

    if (old_xact_count == new_xact_count) {
        ngap_send_ran_ue_context_release_command(ran_ue,
            NGAP_Cause_PR_nas, NGAP_CauseNas_normal_release,
            NGAP_UE_CTX_REL_NG_CONTEXT_REMOVE, 0);
    }
}

void ngap_handle_ue_context_release_request(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    int i;
    char buf[OGS_ADDRSTRLEN];
    uint64_t amf_ue_ngap_id;

    ran_ue_t *ran_ue = NULL;
    amf_ue_t *amf_ue = NULL;
    amf_sess_t *sess = NULL;

    NGAP_InitiatingMessage_t *initiatingMessage = NULL;
    NGAP_UEContextReleaseRequest_t *UEContextReleaseRequest = NULL;

    NGAP_UEContextReleaseRequest_IEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_PDUSessionResourceListCxtRelReq_t *PDUSessionList = NULL;
    NGAP_PDUSessionResourceItemCxtRelReq_t *PDUSessionItem = NULL;
    NGAP_Cause_t *Cause = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    initiatingMessage = message->choice.initiatingMessage;
    ogs_assert(initiatingMessage);
    UEContextReleaseRequest =
        &initiatingMessage->value.choice.UEContextReleaseRequest;
    ogs_assert(UEContextReleaseRequest);

    ogs_debug("UEContextReleaseRequest");

    for (i = 0; i < UEContextReleaseRequest->protocolIEs.list.count; i++) {
        ie = UEContextReleaseRequest->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_PDUSessionResourceListCxtRelReq:
            PDUSessionList = &ie->value.choice.PDUSessionResourceListCxtRelReq;
            break;
        case NGAP_ProtocolIE_ID_id_Cause:
            Cause = &ie->value.choice.Cause;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    ran_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!ran_ue) {
        ogs_warn("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    ogs_debug("    RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld]",
            ran_ue->ran_ue_ngap_id, (long long)ran_ue->amf_ue_ngap_id);

    if (!Cause) {
        ogs_error("No Cause");
        ngap_send_error_indication(
                gnb, &ran_ue->ran_ue_ngap_id, &ran_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    ogs_debug("    Cause[Group:%d Cause:%d]",
            Cause->present, (int)Cause->choice.radioNetwork);

    switch (Cause->present) {
    case NGAP_Cause_PR_radioNetwork:
    case NGAP_Cause_PR_transport:
    case NGAP_Cause_PR_protocol:
    case NGAP_Cause_PR_misc:
        break;
    case NGAP_Cause_PR_nas:
        ogs_warn("NAS-Cause[%d]", (int)Cause->choice.nas);
        break;
    default:
        ogs_warn("Invalid cause group[%d]", Cause->present);
        break;
    }

    amf_ue = ran_ue->amf_ue;
    if (!amf_ue) {
        ogs_error("Cannot find AMF-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_ran_ue_context_release_command(ran_ue,
                Cause->present, (int)Cause->choice.radioNetwork,
                NGAP_UE_CTX_REL_NG_CONTEXT_REMOVE, 0);
    } else {
        int xact_count = amf_sess_xact_count(amf_ue);

        amf_ue->deactivation.group = Cause->present;
        amf_ue->deactivation.cause = (int)Cause->choice.radioNetwork;

        if (!PDUSessionList) {
            amf_sbi_send_deactivate_all_sessions(
                    amf_ue, AMF_UPDATE_SM_CONTEXT_DEACTIVATED,
                    Cause->present, (int)Cause->choice.radioNetwork);
        } else {
            for (i = 0; i < PDUSessionList->list.count; i++) {
                PDUSessionItem = (NGAP_PDUSessionResourceItemCxtRelReq_t *)
                    PDUSessionList->list.array[i];

                if (!PDUSessionItem) {
                    ogs_error("No PDUSessionResourceSetupItemSURes");
                    ngap_send_error_indication2(
                            amf_ue, NGAP_Cause_PR_protocol,
                            NGAP_CauseProtocol_semantic_error);
                    return;
                }

                if (PDUSessionItem->pDUSessionID ==
                        OGS_NAS_PDU_SESSION_IDENTITY_UNASSIGNED) {
                    ogs_error("PDU Session Identity is unassigned");
                    ngap_send_error_indication2(
                            amf_ue, NGAP_Cause_PR_protocol,
                            NGAP_CauseProtocol_semantic_error);
                    return;
                }

                sess = amf_sess_find_by_psi(amf_ue,
                        PDUSessionItem->pDUSessionID);
                if (SESSION_CONTEXT_IN_SMF(sess)) {
                    amf_sbi_send_deactivate_session(
                            sess, AMF_UPDATE_SM_CONTEXT_DEACTIVATED,
                            Cause->present, (int)Cause->choice.radioNetwork);
                }
            }
        }

        if (amf_sess_xact_count(amf_ue) == xact_count)
            ngap_send_amf_ue_context_release_command(amf_ue,
                    Cause->present, (int)Cause->choice.radioNetwork,
                    NGAP_UE_CTX_REL_NG_REMOVE_AND_UNLINK, 0);
    }
}

void ngap_handle_ue_context_release_complete(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    int i;
    char buf[OGS_ADDRSTRLEN];
    uint64_t amf_ue_ngap_id;

    ran_ue_t *ran_ue = NULL;

    NGAP_SuccessfulOutcome_t *successfulOutcome = NULL;
    NGAP_UEContextReleaseComplete_t *UEContextReleaseComplete = NULL;

    NGAP_UEContextReleaseComplete_IEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    successfulOutcome = message->choice.successfulOutcome;
    ogs_assert(successfulOutcome);
    UEContextReleaseComplete =
        &successfulOutcome->value.choice.UEContextReleaseComplete;
    ogs_assert(UEContextReleaseComplete);

    ogs_debug("UEContextReleaseComplete");

    for (i = 0; i < UEContextReleaseComplete->protocolIEs.list.count; i++) {
        ie = UEContextReleaseComplete->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    ran_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!ran_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    ngap_handle_ue_context_release_action(ran_ue);
}

void ngap_handle_ue_context_release_action(ran_ue_t *ran_ue)
{
    amf_ue_t *amf_ue = NULL;

    ogs_assert(ran_ue);

    if (ran_ue_cycle(ran_ue) == NULL) {
        ogs_error("NG context has already been removed");
        return;
    }

    amf_ue = ran_ue->amf_ue;

    ogs_info("UE Context Release [Action:%d]", ran_ue->ue_ctx_rel_action);
    ogs_info("    RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld]",
            ran_ue->ran_ue_ngap_id, (long long)ran_ue->amf_ue_ngap_id);
    if (amf_ue)
        ogs_info("    SUCI[%s]", amf_ue->suci ? amf_ue->suci : "Unknown");

    switch (ran_ue->ue_ctx_rel_action) {
    case NGAP_UE_CTX_REL_NG_CONTEXT_REMOVE:
        ogs_debug("    Action: NG context remove");
        ran_ue_remove(ran_ue);
        break;
    case NGAP_UE_CTX_REL_NG_REMOVE_AND_UNLINK:
        ogs_debug("    Action: NG normal release");
        ran_ue_remove(ran_ue);
        ogs_expect_or_return(amf_ue);
        amf_ue_deassociate(amf_ue);
        break;
    case NGAP_UE_CTX_REL_UE_CONTEXT_REMOVE:
        ogs_debug("    Action: UE context remove");
        ran_ue_remove(ran_ue);
        ogs_expect_or_return(amf_ue);
        amf_ue_remove(amf_ue);
        break;
    case NGAP_UE_CTX_REL_NG_HANDOVER_COMPLETE:
        ogs_debug("    Action: NG handover complete");

        source_ue_deassociate_target_ue(ran_ue);
        ran_ue_remove(ran_ue);

        ogs_expect_or_return(amf_ue);
        break;
    case NGAP_UE_CTX_REL_NG_HANDOVER_CANCEL:
        ogs_warn("    Action: NG handover cancel");

        source_ue_deassociate_target_ue(ran_ue);
        ran_ue_remove(ran_ue);

        ogs_expect_or_return(amf_ue);
        ogs_expect_or_return(amf_ue->ran_ue);

        ngap_send_handover_cancel_ack(amf_ue->ran_ue);
        break;
    case NGAP_UE_CTX_REL_NG_HANDOVER_FAILURE:
        ogs_warn("    Action: NG handover failure");

        source_ue_deassociate_target_ue(ran_ue);
        ran_ue_remove(ran_ue);

        ogs_expect_or_return(amf_ue);
        break;
    default:
        ogs_error("Invalid Action[%d]", ran_ue->ue_ctx_rel_action);
        break;
    }
}

void ngap_handle_pdu_session_resource_setup_response(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    amf_ue_t *amf_ue = NULL;
    ran_ue_t *ran_ue = NULL;
    uint64_t amf_ue_ngap_id;
    amf_nsmf_pdusession_update_sm_context_param_t param;

    NGAP_SuccessfulOutcome_t *successfulOutcome = NULL;
    NGAP_PDUSessionResourceSetupResponse_t *PDUSessionResourceSetupResponse;

    NGAP_PDUSessionResourceSetupResponseIEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_PDUSessionResourceSetupListSURes_t *PDUSessionList = NULL;
    NGAP_PDUSessionResourceSetupItemSURes_t *PDUSessionItem = NULL;
    OCTET_STRING_t *transfer = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    successfulOutcome = message->choice.successfulOutcome;
    ogs_assert(successfulOutcome);
    PDUSessionResourceSetupResponse =
        &successfulOutcome->value.choice.PDUSessionResourceSetupResponse;
    ogs_assert(PDUSessionResourceSetupResponse);

    ogs_debug("PDUSessionResourceSetupResponse");

    for (i = 0; i < PDUSessionResourceSetupResponse->protocolIEs.list.count;
            i++) {
        ie = PDUSessionResourceSetupResponse->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_PDUSessionResourceSetupListSURes:
            PDUSessionList =
                &ie->value.choice.PDUSessionResourceSetupListSURes;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    ran_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!ran_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    ogs_debug("    RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld]",
            ran_ue->ran_ue_ngap_id, (long long)ran_ue->amf_ue_ngap_id);

    amf_ue = ran_ue->amf_ue;
    if (!amf_ue) {
        ogs_error("Cannot find AMF-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &ran_ue->ran_ue_ngap_id, &ran_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    if (!PDUSessionList) {
        ogs_error("No PDUSessionResourceSetupListSURes");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    for (i = 0; i < PDUSessionList->list.count; i++) {
        amf_sess_t *sess = NULL;
        PDUSessionItem = (NGAP_PDUSessionResourceSetupItemSURes_t *)
            PDUSessionList->list.array[i];

        if (!PDUSessionItem) {
            ogs_error("No PDUSessionResourceSetupItemSURes");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        transfer = &PDUSessionItem->pDUSessionResourceSetupResponseTransfer;
        if (!transfer) {
            ogs_error("No PDUSessionResourceSetupResponseTransfer");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        if (PDUSessionItem->pDUSessionID ==
                OGS_NAS_PDU_SESSION_IDENTITY_UNASSIGNED) {
            ogs_error("PDU Session Identity is unassigned");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        sess = amf_sess_find_by_psi(amf_ue, PDUSessionItem->pDUSessionID);
        if (!sess) {
            ogs_error("Cannot find PDU Session ID [%d]",
                    (int)PDUSessionItem->pDUSessionID);
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_radioNetwork,
                    NGAP_CauseRadioNetwork_unknown_PDU_session_ID);
            return;
        }

        if (!SESSION_CONTEXT_IN_SMF(sess)) {
            ogs_error("Session Context is not in SMF [%d]",
                    (int)PDUSessionItem->pDUSessionID);
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_radioNetwork,
                    NGAP_CauseRadioNetwork_unknown_PDU_session_ID);
            return;
        }

        memset(&param, 0, sizeof(param));
        param.n2smbuf = ogs_pkbuf_alloc(NULL, OGS_MAX_SDU_LEN);
        ogs_assert(param.n2smbuf);
        param.n2SmInfoType = OpenAPI_n2_sm_info_type_PDU_RES_SETUP_RSP;
        ogs_pkbuf_put_data(param.n2smbuf, transfer->buf, transfer->size);

        amf_sess_sbi_discover_and_send(OpenAPI_nf_type_SMF,
                sess, AMF_UPDATE_SM_CONTEXT_ACTIVATED, &param,
                amf_nsmf_pdusession_build_update_sm_context);

        ogs_pkbuf_free(param.n2smbuf);
    }
}

void ngap_handle_pdu_session_resource_modify_response(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    amf_ue_t *amf_ue = NULL;
    ran_ue_t *ran_ue = NULL;
    uint64_t amf_ue_ngap_id;
    amf_nsmf_pdusession_update_sm_context_param_t param;

    NGAP_SuccessfulOutcome_t *successfulOutcome = NULL;
    NGAP_PDUSessionResourceModifyResponse_t *PDUSessionResourceModifyResponse;

    NGAP_PDUSessionResourceModifyResponseIEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_PDUSessionResourceModifyListModRes_t *PDUSessionList = NULL;
    NGAP_PDUSessionResourceModifyItemModRes_t *PDUSessionItem = NULL;
    OCTET_STRING_t *transfer = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    successfulOutcome = message->choice.successfulOutcome;
    ogs_assert(successfulOutcome);
    PDUSessionResourceModifyResponse =
        &successfulOutcome->value.choice.PDUSessionResourceModifyResponse;
    ogs_assert(PDUSessionResourceModifyResponse);

    ogs_debug("PDUSessionResourceModifyResponse");

    for (i = 0; i < PDUSessionResourceModifyResponse->protocolIEs.list.count;
            i++) {
        ie = PDUSessionResourceModifyResponse->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_PDUSessionResourceModifyListModRes:
            PDUSessionList =
                &ie->value.choice.PDUSessionResourceModifyListModRes;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    ran_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!ran_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    ogs_debug("    RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld]",
            ran_ue->ran_ue_ngap_id, (long long)ran_ue->amf_ue_ngap_id);

    amf_ue = ran_ue->amf_ue;
    if (!amf_ue) {
        ogs_error("Cannot find AMF-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &ran_ue->ran_ue_ngap_id, &ran_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    if (!PDUSessionList) {
        ogs_error("No PDUSessionResourceModifyListModRes");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    for (i = 0; i < PDUSessionList->list.count; i++) {
        amf_sess_t *sess = NULL;
        PDUSessionItem = (NGAP_PDUSessionResourceModifyItemModRes_t *)
            PDUSessionList->list.array[i];

        if (!PDUSessionItem) {
            ogs_error("No PDUSessionResourceModifyItemModRes");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        transfer = &PDUSessionItem->pDUSessionResourceModifyResponseTransfer;
        if (!transfer) {
            ogs_error("No PDUSessionResourceModifyResponseTransfer");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        if (PDUSessionItem->pDUSessionID ==
                OGS_NAS_PDU_SESSION_IDENTITY_UNASSIGNED) {
            ogs_error("PDU Session Identity is unassigned");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        sess = amf_sess_find_by_psi(amf_ue, PDUSessionItem->pDUSessionID);
        if (!sess) {
            ogs_error("Cannot find PDU Session ID [%d]",
                    (int)PDUSessionItem->pDUSessionID);
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_radioNetwork,
                    NGAP_CauseRadioNetwork_unknown_PDU_session_ID);
            return;
        }

        if (!SESSION_CONTEXT_IN_SMF(sess)) {
            ogs_error("Session Context is not in SMF [%d]",
                    (int)PDUSessionItem->pDUSessionID);
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_radioNetwork,
                    NGAP_CauseRadioNetwork_unknown_PDU_session_ID);
            return;
        }

        memset(&param, 0, sizeof(param));
        param.n2smbuf = ogs_pkbuf_alloc(NULL, OGS_MAX_SDU_LEN);
        ogs_assert(param.n2smbuf);
        param.n2SmInfoType = OpenAPI_n2_sm_info_type_PDU_RES_MOD_RSP;
        ogs_pkbuf_put_data(param.n2smbuf, transfer->buf, transfer->size);

        amf_sess_sbi_discover_and_send(OpenAPI_nf_type_SMF,
                sess, AMF_UPDATE_SM_CONTEXT_MODIFIED, &param,
                amf_nsmf_pdusession_build_update_sm_context);

        ogs_pkbuf_free(param.n2smbuf);
    }
}

void ngap_handle_pdu_session_resource_release_response(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    amf_ue_t *amf_ue = NULL;
    ran_ue_t *ran_ue = NULL;
    uint64_t amf_ue_ngap_id;
    amf_nsmf_pdusession_update_sm_context_param_t param;

    NGAP_SuccessfulOutcome_t *successfulOutcome = NULL;
    NGAP_PDUSessionResourceReleaseResponse_t
        *PDUSessionResourceReleaseResponse;

    NGAP_PDUSessionResourceReleaseResponseIEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_PDUSessionResourceReleasedListRelRes_t *PDUSessionList = NULL;
    NGAP_PDUSessionResourceReleasedItemRelRes_t *PDUSessionItem = NULL;
    OCTET_STRING_t *transfer = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    successfulOutcome = message->choice.successfulOutcome;
    ogs_assert(successfulOutcome);
    PDUSessionResourceReleaseResponse =
        &successfulOutcome->value.choice.PDUSessionResourceReleaseResponse;
    ogs_assert(PDUSessionResourceReleaseResponse);

    ogs_debug("PDUSessionResourceReleaseResponse");

    for (i = 0; i < PDUSessionResourceReleaseResponse->protocolIEs.list.count;
            i++) {
        ie = PDUSessionResourceReleaseResponse->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_PDUSessionResourceReleasedListRelRes:
            PDUSessionList =
                &ie->value.choice.PDUSessionResourceReleasedListRelRes;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    ran_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!ran_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    ogs_debug("    RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld]",
            ran_ue->ran_ue_ngap_id, (long long)ran_ue->amf_ue_ngap_id);

    amf_ue = ran_ue->amf_ue;
    if (!amf_ue) {
        ogs_error("Cannot find AMF-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &ran_ue->ran_ue_ngap_id, &ran_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    if (!PDUSessionList) {
        ogs_error("No PDUSessionResourceReleasedListRelRes");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    for (i = 0; i < PDUSessionList->list.count; i++) {
        amf_sess_t *sess = NULL;
        PDUSessionItem = (NGAP_PDUSessionResourceReleasedItemRelRes_t *)
            PDUSessionList->list.array[i];

        if (!PDUSessionItem) {
            ogs_error("No PDUSessionResourceReleasedItemRelRes");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        transfer = &PDUSessionItem->pDUSessionResourceReleaseResponseTransfer;
        if (!transfer) {
            ogs_error("No PDUSessionResourceReleaseResponseTransfer");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        if (PDUSessionItem->pDUSessionID ==
                OGS_NAS_PDU_SESSION_IDENTITY_UNASSIGNED) {
            ogs_error("PDU Session Identity is unassigned");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        sess = amf_sess_find_by_psi(amf_ue, PDUSessionItem->pDUSessionID);
        if (!sess) {
            ogs_error("Cannot find PDU Session ID [%d]",
                    (int)PDUSessionItem->pDUSessionID);
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_radioNetwork,
                    NGAP_CauseRadioNetwork_unknown_PDU_session_ID);
            return;
        }

        if (!SESSION_CONTEXT_IN_SMF(sess)) {
            ogs_error("Session Context is not in SMF [%d]",
                    (int)PDUSessionItem->pDUSessionID);
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_radioNetwork,
                    NGAP_CauseRadioNetwork_unknown_PDU_session_ID);
            return;
        }

        memset(&param, 0, sizeof(param));
        param.n2smbuf = ogs_pkbuf_alloc(NULL, OGS_MAX_SDU_LEN);
        ogs_assert(param.n2smbuf);
        param.n2SmInfoType = OpenAPI_n2_sm_info_type_PDU_RES_REL_RSP;
        ogs_pkbuf_put_data(param.n2smbuf, transfer->buf, transfer->size);

        amf_sess_sbi_discover_and_send(OpenAPI_nf_type_SMF,
                sess, AMF_UPDATE_SM_CONTEXT_N2_RELEASED, &param,
                amf_nsmf_pdusession_build_update_sm_context);

        ogs_pkbuf_free(param.n2smbuf);
    }
}

void ngap_handle_uplink_ran_configuration_transfer(
        amf_gnb_t *gnb, ogs_ngap_message_t *message, ogs_pkbuf_t *pkbuf)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    NGAP_InitiatingMessage_t *initiatingMessage = NULL;
    NGAP_UplinkRANConfigurationTransfer_t
        *UplinkRANConfigurationTransfer = NULL;

    NGAP_UplinkRANConfigurationTransferIEs_t *ie = NULL;
    NGAP_SONConfigurationTransfer_t *SONConfigurationTransfer = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    initiatingMessage = message->choice.initiatingMessage;
    ogs_assert(initiatingMessage);
    UplinkRANConfigurationTransfer =
        &initiatingMessage->value.choice.UplinkRANConfigurationTransfer;
    ogs_assert(UplinkRANConfigurationTransfer);

    ogs_debug("UplinkRANConfigurationTransfer");
    for (i = 0;
            i < UplinkRANConfigurationTransfer->protocolIEs.list.count; i++) {
        ie = UplinkRANConfigurationTransfer->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_SONConfigurationTransferUL:
            SONConfigurationTransfer =
                &ie->value.choice.SONConfigurationTransfer;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] ENB_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (SONConfigurationTransfer) {
        NGAP_TargetRANNodeID_t *targetRANNodeID = NULL;
        NGAP_GlobalRANNodeID_t *targetGlobalRANNodeID;
        NGAP_GlobalGNB_ID_t *targetGlobalGNB_ID;
        NGAP_SourceRANNodeID_t *sourceRANNodeID = NULL;
        NGAP_GlobalRANNodeID_t *sourceGlobalRANNodeID;
        NGAP_GlobalGNB_ID_t *sourceGlobalGNB_ID;

        amf_gnb_t *target_gnb = NULL;
        uint32_t target_gnb_id, source_gnb_id;
        ogs_5gs_tai_t target_tai, source_tai;

        targetRANNodeID = &SONConfigurationTransfer->targetRANNodeID;
        targetGlobalRANNodeID = &targetRANNodeID->globalRANNodeID;

        if (targetGlobalRANNodeID->present !=
                NGAP_GlobalRANNodeID_PR_globalGNB_ID) {
            ogs_error("Not implemented targetGlobalRANNodeID->present[%d]",
                    targetGlobalRANNodeID->present);
            ngap_send_error_indication(gnb, NULL, NULL,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        targetGlobalGNB_ID = targetGlobalRANNodeID->choice.globalGNB_ID;
        if (!targetGlobalGNB_ID) {
            ogs_error("No targetGlobalGNB_ID");
            ngap_send_error_indication(gnb, NULL, NULL,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        ogs_ngap_GNB_ID_to_uint32(&targetGlobalGNB_ID->gNB_ID, &target_gnb_id);
        ogs_ngap_ASN_to_5gs_tai(&targetRANNodeID->selectedTAI, &target_tai);

        sourceRANNodeID = &SONConfigurationTransfer->sourceRANNodeID;
        sourceGlobalRANNodeID = &sourceRANNodeID->globalRANNodeID;

        if (sourceGlobalRANNodeID->present !=
                NGAP_GlobalRANNodeID_PR_globalGNB_ID) {
            ogs_error("Not implemented sourceGlobalRANNodeID->present[%d]",
                    sourceGlobalRANNodeID->present);
            ngap_send_error_indication(gnb, NULL, NULL,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        sourceGlobalGNB_ID = sourceGlobalRANNodeID->choice.globalGNB_ID;
        if (!sourceGlobalGNB_ID) {
            ogs_error("No sourceGlobalGNB_ID");
            ngap_send_error_indication(gnb, NULL, NULL,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        ogs_ngap_GNB_ID_to_uint32(&sourceGlobalGNB_ID->gNB_ID, &source_gnb_id);
        ogs_ngap_ASN_to_5gs_tai(&sourceRANNodeID->selectedTAI, &source_tai);

        ogs_debug("    Target : GNB_ID[0x%x], TAC[%d]",
                target_gnb_id, target_tai.tac.v);
        ogs_debug("    Source : GNB_ID[0x%x], TAC[%d]",
                source_gnb_id, source_tai.tac.v);

        target_gnb = amf_gnb_find_by_gnb_id(target_gnb_id);
        if (!target_gnb) {
            ogs_error("Uplink RAN configuration transfer : "
                    "cannot find target gNB-id[0x%x]", target_gnb_id);
            ngap_send_error_indication(gnb, NULL, NULL,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        ngap_send_downlink_ran_configuration_transfer(
                target_gnb, SONConfigurationTransfer);
    }
}

void ngap_handle_path_switch_request(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    amf_ue_t *amf_ue = NULL;
    ran_ue_t *ran_ue = NULL;
    uint64_t amf_ue_ngap_id;

    NGAP_InitiatingMessage_t *initiatingMessage = NULL;
    NGAP_PathSwitchRequest_t *PathSwitchRequest = NULL;
    
    NGAP_PathSwitchRequestIEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_UserLocationInformation_t *UserLocationInformation = NULL;
    NGAP_UserLocationInformationNR_t *UserLocationInformationNR = NULL;
    NGAP_UESecurityCapabilities_t *UESecurityCapabilities = NULL;
    NGAP_PDUSessionResourceToBeSwitchedDLList_t
        *PDUSessionResourceToBeSwitchedDLList = NULL;

    NGAP_NRencryptionAlgorithms_t *nRencryptionAlgorithms = NULL;
    NGAP_NRintegrityProtectionAlgorithms_t
        *nRintegrityProtectionAlgorithms = NULL;
    NGAP_EUTRAencryptionAlgorithms_t *eUTRAencryptionAlgorithms = NULL;
    NGAP_EUTRAintegrityProtectionAlgorithms_t
        *eUTRAintegrityProtectionAlgorithms = NULL;
    uint16_t nr_ea = 0, nr_ia = 0, eutra_ea = 0, eutra_ia = 0;
    uint8_t nr_ea0 = 0, nr_ia0 = 0, eutra_ea0 = 0, eutra_ia0 = 0;

    NGAP_PDUSessionResourceToBeSwitchedDLItem_t *PDUSessionItem = NULL;
    OCTET_STRING_t *transfer = NULL;

    amf_nsmf_pdusession_update_sm_context_param_t param;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    initiatingMessage = message->choice.initiatingMessage;
    ogs_assert(initiatingMessage);
    PathSwitchRequest = &initiatingMessage->value.choice.PathSwitchRequest;
    ogs_assert(PathSwitchRequest);

    ogs_info("PathSwitchRequest");
    
    for (i = 0; i < PathSwitchRequest->protocolIEs.list.count; i++) {
        ie = PathSwitchRequest->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_SourceAMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_UserLocationInformation:
            UserLocationInformation = &ie->value.choice.UserLocationInformation;
            break;
        case NGAP_ProtocolIE_ID_id_UESecurityCapabilities:
            UESecurityCapabilities = &ie->value.choice.UESecurityCapabilities;
            break;
        case NGAP_ProtocolIE_ID_id_PDUSessionResourceToBeSwitchedDLList:
            PDUSessionResourceToBeSwitchedDLList =
                &ie->value.choice.PDUSessionResourceToBeSwitchedDLList;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!RAN_UE_NGAP_ID) {
        ogs_error("No RAN_UE_NGAP_ID");
        ngap_send_error_indication(gnb, NULL, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    ran_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!ran_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    amf_ue = ran_ue->amf_ue;
    if (!amf_ue) {
        ogs_error("Cannot find AMF-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &ran_ue->ran_ue_ngap_id, &ran_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    ogs_info("    [OLD] RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] ",
        ran_ue->ran_ue_ngap_id, (long long)ran_ue->amf_ue_ngap_id);
    ogs_info("    [OLD] TAC[%d] CellID[0x%llx]",
        amf_ue->nr_tai.tac.v, (long long)amf_ue->nr_cgi.cell_id);

    /* Update RAN-UE-NGAP-ID */
    ran_ue->ran_ue_ngap_id = *RAN_UE_NGAP_ID;

    /* Change ran_ue to the NEW gNB */
    ran_ue_switch_to_gnb(ran_ue, gnb);

    if (!UserLocationInformation) {
        ogs_error("No UserLocationInformation");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (UserLocationInformation->present !=
            NGAP_UserLocationInformation_PR_userLocationInformationNR) {
        ogs_error("Not implemented UserLocationInformation[%d]",
                UserLocationInformation->present);
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_unspecified);
        return;
    }

    ogs_info("    [NEW] RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] ",
        ran_ue->ran_ue_ngap_id, (long long)ran_ue->amf_ue_ngap_id);

    UserLocationInformationNR =
            UserLocationInformation->choice.userLocationInformationNR;
    ogs_assert(UserLocationInformationNR);
    ogs_ngap_ASN_to_nr_cgi(
            &UserLocationInformationNR->nR_CGI, &ran_ue->saved.nr_cgi);
    ogs_ngap_ASN_to_5gs_tai(
            &UserLocationInformationNR->tAI, &ran_ue->saved.nr_tai);

    /* Copy Stream-No/TAI/ECGI from ran_ue */
    amf_ue->gnb_ostream_id = ran_ue->gnb_ostream_id;
    memcpy(&amf_ue->nr_tai, &ran_ue->saved.nr_tai, sizeof(ogs_5gs_tai_t));
    memcpy(&amf_ue->nr_cgi, &ran_ue->saved.nr_cgi, sizeof(ogs_nr_cgi_t));

    ogs_info("    [NEW] TAC[%d] CellID[0x%llx]",
        amf_ue->nr_tai.tac.v, (long long)amf_ue->nr_cgi.cell_id);

    if (!UESecurityCapabilities) {
        ogs_error("No UESecurityCapabilities");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    nRencryptionAlgorithms = &UESecurityCapabilities->nRencryptionAlgorithms;
    nRintegrityProtectionAlgorithms =
        &UESecurityCapabilities->nRintegrityProtectionAlgorithms;
    eUTRAencryptionAlgorithms =
        &UESecurityCapabilities->eUTRAencryptionAlgorithms;
    eUTRAintegrityProtectionAlgorithms =
        &UESecurityCapabilities->eUTRAintegrityProtectionAlgorithms;

    memcpy(&nr_ea, nRencryptionAlgorithms->buf, sizeof(nr_ea));
    nr_ea = be16toh(nr_ea);
    nr_ea0 = amf_ue->ue_security_capability.nr_ea0;
    amf_ue->ue_security_capability.nr_ea = nr_ea >> 9;
    amf_ue->ue_security_capability.nr_ea0 = nr_ea0;

    memcpy(&nr_ia, nRintegrityProtectionAlgorithms->buf, sizeof(nr_ia));
    nr_ia = be16toh(nr_ia);
    nr_ia0 = amf_ue->ue_security_capability.nr_ia0;
    amf_ue->ue_security_capability.nr_ia = nr_ia >> 9;
    amf_ue->ue_security_capability.nr_ia0 = nr_ia0;

    memcpy(&eutra_ea, eUTRAencryptionAlgorithms->buf, sizeof(eutra_ea));
    eutra_ea = be16toh(eutra_ea);
    eutra_ea0 = amf_ue->ue_security_capability.eutra_ea0;
    amf_ue->ue_security_capability.eutra_ea = eutra_ea >> 9;
    amf_ue->ue_security_capability.eutra_ea0 = eutra_ea0;

    memcpy(&eutra_ia,
            eUTRAintegrityProtectionAlgorithms->buf, sizeof(eutra_ia));
    eutra_ia = be16toh(eutra_ia);
    eutra_ia0 = amf_ue->ue_security_capability.eutra_ia0;
    amf_ue->ue_security_capability.eutra_ia = eutra_ia >> 9;
    amf_ue->ue_security_capability.eutra_ia0 = eutra_ia0;

    if (!SECURITY_CONTEXT_IS_VALID(amf_ue)) {
        ogs_error("No Security Context");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_nas, NGAP_CauseNas_authentication_failure);
        return;
    }

    /* Update Security Context (NextHop) */
    amf_ue->nhcc++;
    ogs_kdf_nh_gnb(amf_ue->kamf, amf_ue->nh, amf_ue->nh);

    if (!PDUSessionResourceToBeSwitchedDLList) {
        ogs_error("No PDUSessionResourceToBeSwitchedDLList");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    for (i = 0; i < PDUSessionResourceToBeSwitchedDLList->list.count; i++) {
        amf_sess_t *sess = NULL;
        PDUSessionItem = (NGAP_PDUSessionResourceToBeSwitchedDLItem_t *)
            PDUSessionResourceToBeSwitchedDLList->list.array[i];

        if (!PDUSessionItem) {
            ogs_error("No NGAP_PDUSessionResourceToBeSwitchedDLItem");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        transfer = &PDUSessionItem->pathSwitchRequestTransfer;
        if (!transfer) {
            ogs_error("No PDUSessionResourceSetupResponseTransfer");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        if (PDUSessionItem->pDUSessionID ==
                OGS_NAS_PDU_SESSION_IDENTITY_UNASSIGNED) {
            ogs_error("PDU Session Identity is unassigned");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        sess = amf_sess_find_by_psi(amf_ue, PDUSessionItem->pDUSessionID);
        if (!sess) {
            ogs_error("Cannot find PDU Session ID [%d]",
                    (int)PDUSessionItem->pDUSessionID);
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_radioNetwork,
                    NGAP_CauseRadioNetwork_unknown_PDU_session_ID);
            return;
        }

        if (!SESSION_CONTEXT_IN_SMF(sess)) {
            ogs_error("Session Context is not in SMF [%d]",
                    (int)PDUSessionItem->pDUSessionID);
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_radioNetwork,
                    NGAP_CauseRadioNetwork_unknown_PDU_session_ID);
            return;
        }

        memset(&param, 0, sizeof(param));
        param.n2smbuf = ogs_pkbuf_alloc(NULL, OGS_MAX_SDU_LEN);
        ogs_assert(param.n2smbuf);
        param.n2SmInfoType = OpenAPI_n2_sm_info_type_PATH_SWITCH_REQ;
        ogs_pkbuf_put_data(param.n2smbuf, transfer->buf, transfer->size);

        amf_sess_sbi_discover_and_send(OpenAPI_nf_type_SMF,
                sess, AMF_UPDATE_SM_CONTEXT_PATH_SWITCH_REQUEST, &param,
                amf_nsmf_pdusession_build_update_sm_context);

        ogs_pkbuf_free(param.n2smbuf);
    }
}

void ngap_handle_handover_required(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    amf_ue_t *amf_ue = NULL;
    ran_ue_t *source_ue = NULL, *target_ue = NULL;
    uint64_t amf_ue_ngap_id;

    amf_gnb_t *target_gnb = NULL;
    uint32_t target_gnb_id;

    NGAP_InitiatingMessage_t *initiatingMessage = NULL;
    NGAP_HandoverRequired_t *HandoverRequired = NULL;

    NGAP_HandoverRequiredIEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_HandoverType_t *HandoverType = NULL;
    NGAP_Cause_t *Cause = NULL;
    NGAP_TargetID_t *TargetID = NULL;
    NGAP_TargetRANNodeID_t *targetRANNodeID = NULL;
    NGAP_GlobalRANNodeID_t *globalRANNodeID = NULL;
    NGAP_GlobalGNB_ID_t *globalGNB_ID = NULL;
    NGAP_PDUSessionResourceListHORqd_t *PDUSessionList = NULL;
    NGAP_PDUSessionResourceItemHORqd_t *PDUSessionItem = NULL;
    OCTET_STRING_t *transfer = NULL;

    NGAP_SourceToTarget_TransparentContainer_t
        *SourceToTarget_TransparentContainer = NULL;

    amf_nsmf_pdusession_update_sm_context_param_t param;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    initiatingMessage = message->choice.initiatingMessage;
    ogs_assert(initiatingMessage);
    HandoverRequired = &initiatingMessage->value.choice.HandoverRequired;
    ogs_assert(HandoverRequired);

    ogs_info("HandoverRequired");

    for (i = 0; i < HandoverRequired->protocolIEs.list.count; i++) {
        ie = HandoverRequired->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_HandoverType:
            HandoverType = &ie->value.choice.HandoverType;
            break;
        case NGAP_ProtocolIE_ID_id_Cause:
            Cause = &ie->value.choice.Cause;
            break;
        case NGAP_ProtocolIE_ID_id_TargetID:
            TargetID = &ie->value.choice.TargetID;
            break;
        case NGAP_ProtocolIE_ID_id_PDUSessionResourceListHORqd:
            PDUSessionList = &ie->value.choice.PDUSessionResourceListHORqd;
            break;
        case NGAP_ProtocolIE_ID_id_SourceToTarget_TransparentContainer:
            SourceToTarget_TransparentContainer =
                &ie->value.choice.SourceToTarget_TransparentContainer;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    source_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!source_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    ogs_debug("    Source : RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] ",
        source_ue->ran_ue_ngap_id, (long long)source_ue->amf_ue_ngap_id);

    amf_ue = source_ue->amf_ue;
    if (!amf_ue) {
        ogs_error("Cannot find AMF-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &source_ue->ran_ue_ngap_id, &source_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    if (!HandoverType) {
        ogs_error("No HandoverType");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (!Cause) {
        ogs_error("No Cause");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (!TargetID) {
        ogs_error("No TargetID");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (TargetID->present != NGAP_TargetID_PR_targetRANNodeID) {
        ogs_error("Not implemented TargetID[%d]", TargetID->present);
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }
    targetRANNodeID = TargetID->choice.targetRANNodeID;
    if (!targetRANNodeID) {
        ogs_error("No targetRANNodeID");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    globalRANNodeID = &targetRANNodeID->globalRANNodeID;
    if (globalRANNodeID->present != NGAP_GlobalRANNodeID_PR_globalGNB_ID) {
        ogs_error("Not implemented globalRANNodeID[%d]",
                globalRANNodeID->present);
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    globalGNB_ID = globalRANNodeID->choice.globalGNB_ID;
    if (!globalGNB_ID) {
        ogs_error("No globalGNB_ID");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    ogs_ngap_GNB_ID_to_uint32(&globalGNB_ID->gNB_ID, &target_gnb_id);
    target_gnb = amf_gnb_find_by_gnb_id(target_gnb_id);
    if (!target_gnb) {
        ogs_error("Handover required : cannot find target gNB-id[0x%x]",
                target_gnb_id);
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (!PDUSessionList) {
        ogs_error("No PDUSessionList");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (!SourceToTarget_TransparentContainer) {
        ogs_error("No SourceToTarget_TransparentContainer");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (!SECURITY_CONTEXT_IS_VALID(amf_ue)) {
        ogs_error("No Security Context");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_nas, NGAP_CauseNas_authentication_failure);
        return;
    }

    /* Target UE */
    target_ue = ran_ue_add(target_gnb, INVALID_UE_NGAP_ID);
    ogs_assert(target_ue);

    /* Source UE - Target UE associated */
    source_ue_associate_target_ue(source_ue, target_ue);

    /* Context Transfer */
    target_ue->ue_context_requested = source_ue->ue_context_requested;
    target_ue->initial_context_setup_request_sent =
            source_ue->initial_context_setup_request_sent;

    ogs_debug("    Target : RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] ",
        target_ue->ran_ue_ngap_id, (long long)target_ue->amf_ue_ngap_id);

    /* Store HandoverType */
    amf_ue->handover.type = *HandoverType;

    /* Store Cause */
    amf_ue->handover.group = Cause->present;
    amf_ue->handover.cause = (int)Cause->choice.radioNetwork;

    /* Update Security Context (NextHop) */
    amf_ue->nhcc++;
    ogs_kdf_nh_gnb(amf_ue->kamf, amf_ue->nh, amf_ue->nh);

    /* Store Container */
    OGS_ASN_STORE_DATA(&amf_ue->handover.container,
            SourceToTarget_TransparentContainer);

    for (i = 0; i < PDUSessionList->list.count; i++) {
        amf_sess_t *sess = NULL;
        PDUSessionItem = (NGAP_PDUSessionResourceItemHORqd_t *)
            PDUSessionList->list.array[i];

        if (!PDUSessionItem) {
            ogs_error("No PDUSessionResourceItemHORqd");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        transfer = &PDUSessionItem->handoverRequiredTransfer;
        if (!transfer) {
            ogs_error("No handoverRequiredTransfer");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        if (PDUSessionItem->pDUSessionID ==
                OGS_NAS_PDU_SESSION_IDENTITY_UNASSIGNED) {
            ogs_error("PDU Session Identity is unassigned");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        sess = amf_sess_find_by_psi(amf_ue, PDUSessionItem->pDUSessionID);
        if (!sess) {
            ogs_error("Cannot find PDU Session ID [%d]",
                    (int)PDUSessionItem->pDUSessionID);
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_radioNetwork,
                    NGAP_CauseRadioNetwork_unknown_PDU_session_ID);
            return;
        }

        if (!SESSION_CONTEXT_IN_SMF(sess)) {
            ogs_error("Session Context is not in SMF [%d]",
                    (int)PDUSessionItem->pDUSessionID);
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_radioNetwork,
                    NGAP_CauseRadioNetwork_unknown_PDU_session_ID);
            return;
        }

        memset(&param, 0, sizeof(param));
        param.n2smbuf = ogs_pkbuf_alloc(NULL, OGS_MAX_SDU_LEN);
        ogs_assert(param.n2smbuf);
        param.n2SmInfoType = OpenAPI_n2_sm_info_type_HANDOVER_REQUIRED;
        ogs_pkbuf_put_data(param.n2smbuf, transfer->buf, transfer->size);

        param.hoState = OpenAPI_ho_state_PREPARING;
        param.TargetID = TargetID;

        amf_sess_sbi_discover_and_send(OpenAPI_nf_type_SMF,
                sess, AMF_UPDATE_SM_CONTEXT_HANDOVER_REQUIRED, &param,
                amf_nsmf_pdusession_build_update_sm_context);

        ogs_pkbuf_free(param.n2smbuf);
    }
}

void ngap_handle_handover_request_ack(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    amf_ue_t *amf_ue = NULL;
    ran_ue_t *source_ue = NULL, *target_ue = NULL;
    uint64_t amf_ue_ngap_id;
    amf_nsmf_pdusession_update_sm_context_param_t param;

    NGAP_SuccessfulOutcome_t *successfulOutcome = NULL;
    NGAP_HandoverRequestAcknowledge_t *HandoverRequestAcknowledge = NULL;

    NGAP_HandoverRequestAcknowledgeIEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_PDUSessionResourceAdmittedList_t *PDUSessionList = NULL;
    NGAP_PDUSessionResourceAdmittedItem_t *PDUSessionItem = NULL;
    OCTET_STRING_t *transfer = NULL;

    NGAP_TargetToSource_TransparentContainer_t
        *TargetToSource_TransparentContainer = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    successfulOutcome = message->choice.successfulOutcome;
    ogs_assert(successfulOutcome);
    HandoverRequestAcknowledge =
        &successfulOutcome->value.choice.HandoverRequestAcknowledge;
    ogs_assert(HandoverRequestAcknowledge);

    ogs_debug("HandoverRequestAcknowledge");

    for (i = 0; i < HandoverRequestAcknowledge->protocolIEs.list.count; i++) {
        ie = HandoverRequestAcknowledge->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_PDUSessionResourceAdmittedList:
            PDUSessionList = &ie->value.choice.PDUSessionResourceAdmittedList;
            break;
        case NGAP_ProtocolIE_ID_id_TargetToSource_TransparentContainer:
            TargetToSource_TransparentContainer =
                &ie->value.choice.TargetToSource_TransparentContainer;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!RAN_UE_NGAP_ID) {
        ogs_error("No RAN_UE_NGAP_ID");
        ngap_send_error_indication(gnb, NULL, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    target_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!target_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    if (!RAN_UE_NGAP_ID) {
        ogs_error("No RAN_UE_NGAP_ID");
        ngap_send_error_indication(gnb, NULL, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    target_ue->ran_ue_ngap_id = *RAN_UE_NGAP_ID;

    source_ue = target_ue->source_ue;
    if (!source_ue) {
        ogs_error("Cannot find Source-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &target_ue->ran_ue_ngap_id, &target_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_inconsistent_remote_UE_NGAP_ID);
        return;
    }
    amf_ue = target_ue->amf_ue;
    if (!amf_ue) {
        ogs_error("Cannot find AMF-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &target_ue->ran_ue_ngap_id, &target_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    ogs_debug("    Source : RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] ",
        source_ue->ran_ue_ngap_id, (long long)source_ue->amf_ue_ngap_id);
    ogs_debug("    Target : RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] ",
        target_ue->ran_ue_ngap_id, (long long)target_ue->amf_ue_ngap_id);

    if (!PDUSessionList) {
        ogs_error("No PDUSessionList");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (!TargetToSource_TransparentContainer) {
        ogs_error("No TargetToSource_TransparentContainer");
        ngap_send_error_indication2(amf_ue,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    /* Store Container */
    OGS_ASN_STORE_DATA(&amf_ue->handover.container,
            TargetToSource_TransparentContainer);

    for (i = 0; i < PDUSessionList->list.count; i++) {
        amf_sess_t *sess = NULL;
        PDUSessionItem = (NGAP_PDUSessionResourceAdmittedItem_t *)
            PDUSessionList->list.array[i];

        if (!PDUSessionItem) {
            ogs_error("No PDUSessionResourceAdmittedItem");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        transfer = &PDUSessionItem->handoverRequestAcknowledgeTransfer;
        if (!transfer) {
            ogs_error("No handoverRequestAcknowledgeTransfer");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        if (PDUSessionItem->pDUSessionID ==
                OGS_NAS_PDU_SESSION_IDENTITY_UNASSIGNED) {
            ogs_error("PDU Session Identity is unassigned");
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
            return;
        }

        sess = amf_sess_find_by_psi(amf_ue, PDUSessionItem->pDUSessionID);
        if (!sess) {
            ogs_error("Cannot find PDU Session ID [%d]",
                    (int)PDUSessionItem->pDUSessionID);
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_radioNetwork,
                    NGAP_CauseRadioNetwork_unknown_PDU_session_ID);
            return;
        }

        if (!SESSION_CONTEXT_IN_SMF(sess)) {
            ogs_error("Session Context is not in SMF [%d]",
                    (int)PDUSessionItem->pDUSessionID);
            ngap_send_error_indication2(amf_ue,
                    NGAP_Cause_PR_radioNetwork,
                    NGAP_CauseRadioNetwork_unknown_PDU_session_ID);
            return;
        }

        memset(&param, 0, sizeof(param));
        param.n2smbuf = ogs_pkbuf_alloc(NULL, OGS_MAX_SDU_LEN);
        ogs_assert(param.n2smbuf);
        param.n2SmInfoType = OpenAPI_n2_sm_info_type_HANDOVER_REQ_ACK;
        ogs_pkbuf_put_data(param.n2smbuf, transfer->buf, transfer->size);

        param.hoState = OpenAPI_ho_state_PREPARED;

        amf_sess_sbi_discover_and_send(OpenAPI_nf_type_SMF,
                sess, AMF_UPDATE_SM_CONTEXT_HANDOVER_REQ_ACK, &param,
                amf_nsmf_pdusession_build_update_sm_context);

        ogs_pkbuf_free(param.n2smbuf);
    }
}

void ngap_handle_handover_failure(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    ran_ue_t *source_ue = NULL, *target_ue = NULL;
    uint64_t amf_ue_ngap_id;

    NGAP_UnsuccessfulOutcome_t *unsuccessfulOutcome = NULL;
    NGAP_HandoverFailure_t *HandoverFailure = NULL;

    NGAP_HandoverFailureIEs_t *ie = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_Cause_t *Cause = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    unsuccessfulOutcome = message->choice.unsuccessfulOutcome;
    ogs_assert(unsuccessfulOutcome);
    HandoverFailure =
        &unsuccessfulOutcome->value.choice.HandoverFailure;
    ogs_assert(HandoverFailure);

    ogs_debug("HandoverFailure");

    for (i = 0; i < HandoverFailure->protocolIEs.list.count; i++) {
        ie = HandoverFailure->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_Cause:
            Cause = &ie->value.choice.Cause;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, NULL, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, NULL, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    target_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!target_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, NULL, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    source_ue = target_ue->source_ue;
    if (!source_ue) {
        ogs_error("Cannot find Source-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &target_ue->ran_ue_ngap_id, &target_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_inconsistent_remote_UE_NGAP_ID);
        return;
    }

    ogs_debug("    Source : RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] ",
        source_ue->ran_ue_ngap_id, (long long)source_ue->amf_ue_ngap_id);
    ogs_debug("    Target : RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] ",
        target_ue->ran_ue_ngap_id, (long long)target_ue->amf_ue_ngap_id);

    if (!Cause) {
        ogs_error("No Cause");
        ngap_send_error_indication(
                gnb, &target_ue->ran_ue_ngap_id, &target_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }
    ogs_debug("    Cause[Group:%d Cause:%d]",
            Cause->present, (int)Cause->choice.radioNetwork);

    ngap_send_handover_preparation_failure(source_ue, Cause);

    ngap_send_ran_ue_context_release_command(target_ue,
            NGAP_Cause_PR_radioNetwork, NGAP_CauseRadioNetwork_ho_failure_in_target_5GC_ngran_node_or_target_system,
            NGAP_UE_CTX_REL_NG_HANDOVER_FAILURE, 0);
}

void ngap_handle_handover_cancel(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    amf_ue_t *amf_ue = NULL;
    amf_sess_t *sess = NULL;
    ran_ue_t *source_ue = NULL, *target_ue = NULL;
    uint64_t amf_ue_ngap_id;
    amf_nsmf_pdusession_update_sm_context_param_t param;

    NGAP_InitiatingMessage_t *initiatingMessage = NULL;
    NGAP_HandoverCancel_t *HandoverCancel = NULL;

    NGAP_HandoverCancelIEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_Cause_t *Cause = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    initiatingMessage = message->choice.initiatingMessage;
    ogs_assert(initiatingMessage);
    HandoverCancel = &initiatingMessage->value.choice.HandoverCancel;
    ogs_assert(HandoverCancel);

    ogs_debug("HandoverCancel");

    for (i = 0; i < HandoverCancel->protocolIEs.list.count; i++) {
        ie = HandoverCancel->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_Cause:
            Cause = &ie->value.choice.Cause;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    source_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!source_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    target_ue = source_ue->target_ue;
    if (!target_ue) {
        ogs_error("Cannot find Source-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &source_ue->ran_ue_ngap_id, &source_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_inconsistent_remote_UE_NGAP_ID);
        return;
    }
    amf_ue = source_ue->amf_ue;
    if (!amf_ue) {
        ogs_error("Cannot find AMF-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &source_ue->ran_ue_ngap_id, &source_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    ogs_debug("    Source : RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] ",
        source_ue->ran_ue_ngap_id, (long long)source_ue->amf_ue_ngap_id);
    ogs_debug("    Target : RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] ",
        target_ue->ran_ue_ngap_id, (long long)target_ue->amf_ue_ngap_id);

    if (!Cause) {
        ogs_error("No Cause");
        ngap_send_error_indication(
                gnb, &source_ue->ran_ue_ngap_id, &source_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }
    ogs_debug("    Cause[Group:%d Cause:%d]",
            Cause->present, (int)Cause->choice.radioNetwork);

    ogs_list_for_each(&amf_ue->sess_list, sess) {
        memset(&param, 0, sizeof(param));
        param.hoState = OpenAPI_ho_state_CANCELLED;
        param.ngApCause.group = Cause->present;
        param.ngApCause.value = (int)Cause->choice.radioNetwork;

        amf_sess_sbi_discover_and_send(OpenAPI_nf_type_SMF,
                sess, AMF_UPDATE_SM_CONTEXT_HANDOVER_CANCEL, &param,
                amf_nsmf_pdusession_build_update_sm_context);
    }
}

void ngap_handle_uplink_ran_status_transfer(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    amf_ue_t *amf_ue = NULL;
    ran_ue_t *source_ue = NULL, *target_ue = NULL;
    uint64_t amf_ue_ngap_id;

    NGAP_InitiatingMessage_t *initiatingMessage = NULL;
    NGAP_UplinkRANStatusTransfer_t *UplinkRANStatusTransfer = NULL;

    NGAP_UplinkRANStatusTransferIEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_RANStatusTransfer_TransparentContainer_t
        *RANStatusTransfer_TransparentContainer = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    initiatingMessage = message->choice.initiatingMessage;
    ogs_assert(initiatingMessage);
    UplinkRANStatusTransfer =
        &initiatingMessage->value.choice.UplinkRANStatusTransfer;
    ogs_assert(UplinkRANStatusTransfer);

    ogs_debug("UplinkRANStatusTransfer");

    for (i = 0; i < UplinkRANStatusTransfer->protocolIEs.list.count; i++) {
        ie = UplinkRANStatusTransfer->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_RANStatusTransfer_TransparentContainer:
            RANStatusTransfer_TransparentContainer =
                &ie->value.choice.RANStatusTransfer_TransparentContainer;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    source_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!source_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    target_ue = source_ue->target_ue;
    if (!target_ue) {
        ogs_error("Cannot find Source-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &source_ue->ran_ue_ngap_id, &source_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_inconsistent_remote_UE_NGAP_ID);
        return;
    }
    amf_ue = source_ue->amf_ue;
    if (!amf_ue) {
        ogs_error("Cannot find AMF-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &source_ue->ran_ue_ngap_id, &source_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    ogs_debug("    Source : RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] ",
        source_ue->ran_ue_ngap_id, (long long)source_ue->amf_ue_ngap_id);
    ogs_debug("    Target : RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] ",
        target_ue->ran_ue_ngap_id, (long long)target_ue->amf_ue_ngap_id);

    ngap_send_downlink_ran_status_transfer(
            target_ue, RANStatusTransfer_TransparentContainer);
}

void ngap_handle_handover_notification(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    amf_ue_t *amf_ue = NULL;
    amf_sess_t *sess = NULL;
    ran_ue_t *source_ue = NULL, *target_ue = NULL;
    uint64_t amf_ue_ngap_id;
    amf_nsmf_pdusession_update_sm_context_param_t param;

    NGAP_InitiatingMessage_t *initiatingMessage = NULL;
    NGAP_HandoverNotify_t *HandoverNotify = NULL;

    NGAP_HandoverNotifyIEs_t *ie = NULL;
    NGAP_RAN_UE_NGAP_ID_t *RAN_UE_NGAP_ID = NULL;
    NGAP_AMF_UE_NGAP_ID_t *AMF_UE_NGAP_ID = NULL;
    NGAP_UserLocationInformation_t *UserLocationInformation = NULL;
    NGAP_UserLocationInformationNR_t *UserLocationInformationNR = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    initiatingMessage = message->choice.initiatingMessage;
    ogs_assert(initiatingMessage);
    HandoverNotify = &initiatingMessage->value.choice.HandoverNotify;
    ogs_assert(HandoverNotify);

    ogs_debug("HandoverNotify");

    for (i = 0; i < HandoverNotify->protocolIEs.list.count; i++) {
        ie = HandoverNotify->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_RAN_UE_NGAP_ID:
            RAN_UE_NGAP_ID = &ie->value.choice.RAN_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_AMF_UE_NGAP_ID:
            AMF_UE_NGAP_ID = &ie->value.choice.AMF_UE_NGAP_ID;
            break;
        case NGAP_ProtocolIE_ID_id_UserLocationInformation:
            UserLocationInformation = &ie->value.choice.UserLocationInformation;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] RAN_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!AMF_UE_NGAP_ID) {
        ogs_error("No AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (asn_INTEGER2ulong(AMF_UE_NGAP_ID,
                (unsigned long *)&amf_ue_ngap_id) != 0) {
        ogs_error("Invalid AMF_UE_NGAP_ID");
        ngap_send_error_indication(gnb, (uint32_t *)RAN_UE_NGAP_ID, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    target_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);
    if (!target_ue) {
        ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, (uint32_t *)RAN_UE_NGAP_ID, &amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    source_ue = target_ue->source_ue;
    if (!source_ue) {
        ogs_error("Cannot find Source-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &target_ue->ran_ue_ngap_id, &target_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_inconsistent_remote_UE_NGAP_ID);
        return;
    }
    amf_ue = target_ue->amf_ue;
    if (!amf_ue) {
        ogs_error("Cannot find AMF-UE Context [%lld]",
                (long long)amf_ue_ngap_id);
        ngap_send_error_indication(
                gnb, &target_ue->ran_ue_ngap_id, &target_ue->amf_ue_ngap_id,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
        return;
    }

    amf_ue_associate_ran_ue(amf_ue, target_ue);

    if (!UserLocationInformation) {
        ogs_error("No UserLocationInformation");
        ngap_send_error_indication(gnb, &target_ue->ran_ue_ngap_id, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    if (UserLocationInformation->present !=
            NGAP_UserLocationInformation_PR_userLocationInformationNR) {
        ogs_error("Not implemented UserLocationInformation[%d]",
                UserLocationInformation->present);
        ngap_send_error_indication(gnb, &target_ue->ran_ue_ngap_id, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_unspecified);
        return;
    }

    UserLocationInformationNR =
        UserLocationInformation->choice.userLocationInformationNR;
    ogs_assert(UserLocationInformationNR);
    ogs_ngap_ASN_to_nr_cgi(
            &UserLocationInformationNR->nR_CGI, &target_ue->saved.nr_cgi);
    ogs_ngap_ASN_to_5gs_tai(
            &UserLocationInformationNR->tAI, &target_ue->saved.nr_tai);

    ogs_debug("    Source : RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] ",
        source_ue->ran_ue_ngap_id, (long long)source_ue->amf_ue_ngap_id);
    ogs_debug("    Source : TAC[%d] CellID[0x%llx]",
        source_ue->saved.nr_tai.tac.v,
        (long long)source_ue->saved.nr_cgi.cell_id);
    ogs_debug("    Target : RAN_UE_NGAP_ID[%d] AMF_UE_NGAP_ID[%lld] ",
        target_ue->ran_ue_ngap_id, (long long)target_ue->amf_ue_ngap_id);
    ogs_debug("    Target : TAC[%d] CellID[0x%llx]",
        target_ue->saved.nr_tai.tac.v,
        (long long)target_ue->saved.nr_cgi.cell_id);

    /* Copy Stream-No/TAI/ECGI from ran_ue */
    amf_ue->gnb_ostream_id = target_ue->gnb_ostream_id;
    memcpy(&amf_ue->nr_tai, &target_ue->saved.nr_tai, sizeof(ogs_5gs_tai_t));
    memcpy(&amf_ue->nr_cgi, &target_ue->saved.nr_cgi, sizeof(ogs_nr_cgi_t));

    ngap_send_ran_ue_context_release_command(source_ue,
            NGAP_Cause_PR_radioNetwork,
            NGAP_CauseRadioNetwork_successful_handover,
            NGAP_UE_CTX_REL_NG_HANDOVER_COMPLETE,
            ogs_app()->time.handover.duration);

    ogs_list_for_each(&amf_ue->sess_list, sess) {
        memset(&param, 0, sizeof(param));
        param.hoState = OpenAPI_ho_state_COMPLETED;

        amf_sess_sbi_discover_and_send(OpenAPI_nf_type_SMF,
                sess, AMF_UPDATE_SM_CONTEXT_HANDOVER_NOTIFY, &param,
                amf_nsmf_pdusession_build_update_sm_context);
    }
}

void ngap_handle_ran_configuration_update(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i, j, k;

    NGAP_InitiatingMessage_t *initiatingMessage = NULL;
    NGAP_RANConfigurationUpdate_t *RANConfigurationUpdate = NULL;

    NGAP_RANConfigurationUpdateIEs_t *ie = NULL;
    NGAP_GlobalRANNodeID_t *GlobalRANNodeID = NULL;
    NGAP_GlobalGNB_ID_t *globalGNB_ID = NULL;
    NGAP_SupportedTAList_t *SupportedTAList = NULL;
    NGAP_PagingDRX_t *PagingDRX = NULL;

    NGAP_Cause_PR group = NGAP_Cause_PR_NOTHING;
    long cause = 0;

    uint32_t gnb_id;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    initiatingMessage = message->choice.initiatingMessage;
    ogs_assert(initiatingMessage);
    RANConfigurationUpdate = &initiatingMessage->value.choice.RANConfigurationUpdate;
    ogs_assert(RANConfigurationUpdate);

    ogs_debug("RANConfigurationUpdate");

    for (i = 0; i < RANConfigurationUpdate->protocolIEs.list.count; i++) {
        ie = RANConfigurationUpdate->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_GlobalRANNodeID:
            GlobalRANNodeID = &ie->value.choice.GlobalRANNodeID;
            break;
        case NGAP_ProtocolIE_ID_id_SupportedTAList:
            SupportedTAList = &ie->value.choice.SupportedTAList;
            break;
        case NGAP_ProtocolIE_ID_id_DefaultPagingDRX:
            PagingDRX = &ie->value.choice.PagingDRX;
            break;
        default:
            break;
        }
    }

    if (GlobalRANNodeID) {
        globalGNB_ID = GlobalRANNodeID->choice.globalGNB_ID;
        if (!globalGNB_ID) {
            ogs_error("No globalGNB_ID");
            group = NGAP_Cause_PR_protocol;
            cause = NGAP_CauseProtocol_semantic_error;
            ngap_send_ran_configuration_update_failure(gnb, group, cause);
            return;
        }

        ogs_ngap_GNB_ID_to_uint32(&globalGNB_ID->gNB_ID, &gnb_id);
        ogs_debug("    IP[%s] GNB_ID[0x%x]",
                OGS_ADDR(gnb->sctp.addr, buf), gnb_id);

        amf_gnb_set_gnb_id(gnb, gnb_id);
    }

    if (SupportedTAList) {
        /* Parse Supported TA */
        for (i = 0, gnb->num_of_supported_ta_list = 0;
                i < SupportedTAList->list.count &&
                gnb->num_of_supported_ta_list < OGS_MAX_NUM_OF_TAI;
                    i++) {
            NGAP_SupportedTAItem_t *SupportedTAItem = NULL;

            SupportedTAItem = (NGAP_SupportedTAItem_t *)
                    SupportedTAList->list.array[i];
            if (!SupportedTAItem) {
                ogs_error("No SupportedTAItem");
                group = NGAP_Cause_PR_protocol;
                cause = NGAP_CauseProtocol_semantic_error;
                ngap_send_ran_configuration_update_failure(gnb, group, cause);
                return;
            }

            ogs_asn_OCTET_STRING_to_uint24(&SupportedTAItem->tAC,
                &gnb->supported_ta_list[i].tac);

            ogs_debug("    TAC[%d]", gnb->supported_ta_list[i].tac.v);

            for (j = 0, gnb->supported_ta_list[i].num_of_bplmn_list = 0;
                    j < SupportedTAItem->broadcastPLMNList.list.count &&
                    gnb->supported_ta_list[i].num_of_bplmn_list <
                        OGS_MAX_NUM_OF_BPLMN;
                            j++) {

                NGAP_BroadcastPLMNItem_t *BroadcastPLMNItem = NULL;
                NGAP_PLMNIdentity_t *pLMNIdentity = NULL;

                BroadcastPLMNItem = (NGAP_BroadcastPLMNItem_t *)
                        SupportedTAItem->broadcastPLMNList.list.array[j];
                if (!BroadcastPLMNItem) {
                    ogs_error("No BroadcastPLMNItem");
                    group = NGAP_Cause_PR_protocol;
                    cause = NGAP_CauseProtocol_semantic_error;
                    ngap_send_ran_configuration_update_failure(
                            gnb, group, cause);
                    return;
                }

                pLMNIdentity = (NGAP_PLMNIdentity_t *)
                        &BroadcastPLMNItem->pLMNIdentity;
                ogs_assert(pLMNIdentity);

                memcpy(&gnb->supported_ta_list[i].bplmn_list[j].plmn_id,
                        pLMNIdentity->buf, sizeof(ogs_plmn_id_t));
                ogs_debug("    PLMN_ID[MCC:%d MNC:%d]",
                    ogs_plmn_id_mcc(&gnb->supported_ta_list[i].
                        bplmn_list[j].plmn_id),
                    ogs_plmn_id_mnc(&gnb->supported_ta_list[i].
                        bplmn_list[j].plmn_id));

                for (k = 0, gnb->supported_ta_list[i].
                                bplmn_list[j].num_of_s_nssai = 0;
                        k < BroadcastPLMNItem->tAISliceSupportList.list.count &&
                        gnb->supported_ta_list[i].bplmn_list[j].num_of_s_nssai <
                            OGS_MAX_NUM_OF_SLICE;
                                k++) {
                    NGAP_SliceSupportItem_t *SliceSupportItem = NULL;
                    NGAP_S_NSSAI_t *s_NSSAI = NULL;

                    SliceSupportItem = (NGAP_SliceSupportItem_t *)
                        BroadcastPLMNItem->tAISliceSupportList.list.array[k];
                    if (!SliceSupportItem) {
                        ogs_error("No SliceSupportItem");
                        group = NGAP_Cause_PR_protocol;
                        cause = NGAP_CauseProtocol_semantic_error;
                        ngap_send_ran_configuration_update_failure(
                                gnb, group, cause);
                        return;
                    }

                    s_NSSAI = &SliceSupportItem->s_NSSAI;
                    ogs_assert(s_NSSAI);

                    ogs_asn_OCTET_STRING_to_uint8(&s_NSSAI->sST,
                        &gnb->supported_ta_list[i].
                            bplmn_list[j].s_nssai[k].sst);
                    if (!s_NSSAI->sD) {
                        gnb->supported_ta_list[i].bplmn_list[j].
                            s_nssai[k].sd.v = OGS_S_NSSAI_NO_SD_VALUE;
                    } else {
                        ogs_asn_OCTET_STRING_to_uint24(s_NSSAI->sD,
                        &gnb->supported_ta_list[i].bplmn_list[j].s_nssai[k].sd);
                    }

                    ogs_debug("    S_NSSAI[SST:%d SD:0x%x]",
                        gnb->supported_ta_list[i].bplmn_list[j].s_nssai[k].sst,
                        gnb->supported_ta_list[i].bplmn_list[j].
                            s_nssai[k].sd.v);

                    gnb->supported_ta_list[i].bplmn_list[j].num_of_s_nssai++;
                }

                gnb->supported_ta_list[i].num_of_bplmn_list++;
            }

            gnb->num_of_supported_ta_list++;
        }

        if (gnb->num_of_supported_ta_list == 0) {
            ogs_warn("RANConfigurationUpdate failure:");
            ogs_warn("    No supported TA exist in request");
            group = NGAP_Cause_PR_protocol;
            cause =
                NGAP_CauseProtocol_message_not_compatible_with_receiver_state;

            ngap_send_ran_configuration_update_failure(gnb, group, cause);
            return;
        }

        if (!served_tai_is_found(gnb)) {
            ogs_warn("RANConfigurationUpdate failure:");
            ogs_warn("    Cannot find Served TAI. "
                        "Check 'amf.tai' configuration");
            group = NGAP_Cause_PR_misc;
            cause = NGAP_CauseMisc_unknown_PLMN;

            ngap_send_ran_configuration_update_failure(gnb, group, cause);
            return;
        }

        if (!s_nssai_is_found(gnb)) {
            ogs_warn("RANConfigurationUpdate failure:");
            ogs_warn("    Cannot find S_NSSAI. "
                        "Check 'amf.plmn_support.s_nssai' configuration");
            group = NGAP_Cause_PR_misc;
            cause = NGAP_CauseMisc_unknown_PLMN;

            ngap_send_ran_configuration_update_failure(gnb, group, cause);
            return;
        }
    }

    if (PagingDRX)
        ogs_debug("    PagingDRX[%ld]", *PagingDRX);

    ngap_send_ran_configuration_update_ack(gnb);
}

void ngap_handle_ng_reset(
        amf_gnb_t *gnb, ogs_ngap_message_t *message)
{
    char buf[OGS_ADDRSTRLEN];
    int i;

    NGAP_InitiatingMessage_t *initiatingMessage = NULL;
    NGAP_NGReset_t *NGReset = NULL;

    NGAP_NGResetIEs_t *ie = NULL;
    NGAP_Cause_t *Cause = NULL;
    NGAP_ResetType_t *ResetType = NULL;
    NGAP_UE_associatedLogicalNG_connectionList_t *partOfNG_Interface = NULL;

    ran_ue_t *iter = NULL;

    ogs_assert(gnb);
    ogs_assert(gnb->sctp.sock);

    ogs_assert(message);
    initiatingMessage = message->choice.initiatingMessage;
    ogs_assert(initiatingMessage);
    NGReset = &initiatingMessage->value.choice.NGReset;
    ogs_assert(NGReset);

    ogs_warn("NGReset");

    for (i = 0; i < NGReset->protocolIEs.list.count; i++) {
        ie = NGReset->protocolIEs.list.array[i];
        switch (ie->id) {
        case NGAP_ProtocolIE_ID_id_Cause:
            Cause = &ie->value.choice.Cause;
            break;
        case NGAP_ProtocolIE_ID_id_ResetType:
            ResetType = &ie->value.choice.ResetType;
            break;
        default:
            break;
        }
    }

    ogs_debug("    IP[%s] ENB_ID[%d]",
            OGS_ADDR(gnb->sctp.addr, buf), gnb->gnb_id);

    if (!Cause) {
        ogs_error("No Cause");
        ngap_send_error_indication(gnb, NULL, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    ogs_warn("    Cause[Group:%d Cause:%d]",
            Cause->present, (int)Cause->choice.radioNetwork);

    if (!ResetType) {
        ogs_error("No ResetType");
        ngap_send_error_indication(gnb, NULL, NULL,
                NGAP_Cause_PR_protocol, NGAP_CauseProtocol_semantic_error);
        return;
    }

    switch (ResetType->present) {
    case NGAP_ResetType_PR_nG_Interface:
        ogs_warn("    NGAP_ResetType_PR_nG_Interface");

        amf_sbi_send_deactivate_all_ue_in_gnb(
                gnb, AMF_REMOVE_S1_CONTEXT_BY_RESET_ALL);

    /*
     * TS38.413
     *
     * 8.7.4.2.1 NG Reset initiated by the AMF
     * At reception of the NG RESET message the NG-RAN node shall release all
     * allocated resources on NG and Uu related to the UE association(s)
     * indicated explicitly or implicitly in the NG RESET message and
     * remove the indicated UE contexts including NGAP ID.
     *
     * After the NG-RAN node has released all assigned NG resources and
     * the UE NGAP IDs for all indicated UE associations which can be used
     * for new UE-associated logical NG-connections over the NG interface,
     * the NG-RAN node shall respond with the NG RESET ACKNOWLEDGE message.
     *
     * The NG-RAN node does not need to wait for the release of radio resources
     * to be completed before returning the NG RESET ACKNOWLEDGE message.
     *
     * 8.7.4.2.2 NG Reset initiated by the NG-RAN node
     *
     * At reception of the NG RESET message the AMF shall release
     * all allocated resources on NG related to the UE association(s)
     * indicated explicitly or implicitly in the NG RESET message and
     * remove the NGAP ID for the indicated UE associations.
     *
     * After the AMF has released all assigned NG resources and
     * the UE NGAP IDs for all indicated UE associations which can be used
     * for new UE-associated logical NG-connections over the NG interface,
     * the AMF shall respond with the NG RESET ACKNOWLEDGE message.
     */
        if (ogs_list_count(&gnb->ran_ue_list) == 0)
            ngap_send_ng_reset_ack(gnb, NULL);

        break;

    case NGAP_ResetType_PR_partOfNG_Interface:
        ogs_warn("    NGAP_ResetType_PR_partOfNG_Interface");

        partOfNG_Interface = ResetType->choice.partOfNG_Interface;
        ogs_assert(partOfNG_Interface);
        for (i = 0; i < partOfNG_Interface->list.count; i++) {
            NGAP_UE_associatedLogicalNG_connectionItem_t *item = NULL;
            uint64_t amf_ue_ngap_id;

            ran_ue_t *ran_ue = NULL;
            amf_ue_t *amf_ue = NULL;

            item = (NGAP_UE_associatedLogicalNG_connectionItem_t *)
                        partOfNG_Interface->list.array[i];
            if (!item) {
                ogs_error("No ResetType");
                ngap_send_error_indication(
                        gnb, NULL, NULL,
                        NGAP_Cause_PR_protocol,
                        NGAP_CauseProtocol_semantic_error);
                return;
            }

            if (item->aMF_UE_NGAP_ID) {
                if (asn_INTEGER2ulong(item->aMF_UE_NGAP_ID,
                            (unsigned long *)&amf_ue_ngap_id) != 0) {
                    ogs_error("Invalid AMF_UE_NGAP_ID");
                    ngap_send_error_indication(
                            gnb, NULL, NULL,
                            NGAP_Cause_PR_protocol,
                            NGAP_CauseProtocol_semantic_error);
                    return;
                }

                ran_ue = ran_ue_find_by_amf_ue_ngap_id(amf_ue_ngap_id);

                if (!ran_ue) {
                    ogs_error("No RAN UE Context : AMF_UE_NGAP_ID[%lld]",
                            (long long)amf_ue_ngap_id);
                    ngap_send_error_indication(
                            gnb, NULL, &amf_ue_ngap_id,
                            NGAP_Cause_PR_radioNetwork,
                            NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
                    return;
                }

            } else if (item->rAN_UE_NGAP_ID) {

                ran_ue = ran_ue_find_by_ran_ue_ngap_id(
                            gnb, *item->rAN_UE_NGAP_ID);

                if (!ran_ue) {
                    ogs_error("No RAN UE Context : RAN_UE_NGAP_ID[%d]",
                            (int)*item->rAN_UE_NGAP_ID);
                    ngap_send_error_indication(
                            gnb, NULL, NULL,
                            NGAP_Cause_PR_radioNetwork,
                            NGAP_CauseRadioNetwork_unknown_local_UE_NGAP_ID);
                    return;
                }
            } else {
                ogs_error("No UE NGAP ID");
                ngap_send_error_indication(
                        gnb, NULL, NULL,
                        NGAP_Cause_PR_protocol,
                        NGAP_CauseProtocol_semantic_error);
                return;
            }

            ogs_assert(ran_ue);

            /* RAN_UE Context where PartOfNG_interface was requested */
            ran_ue->part_of_ng_reset_requested = true;

            amf_ue = ran_ue->amf_ue;
            ogs_assert(amf_ue);

            amf_sbi_send_deactivate_all_sessions(
                amf_ue, AMF_REMOVE_S1_CONTEXT_BY_RESET_PARTIAL,
                NGAP_Cause_PR_radioNetwork,
                NGAP_CauseRadioNetwork_failure_in_radio_interface_procedure);
        }

        if (gnb->ng_reset_ack)
            ogs_pkbuf_free(gnb->ng_reset_ack);

        gnb->ng_reset_ack = ogs_ngap_build_ng_reset_ack(partOfNG_Interface);
        ogs_expect_or_return(gnb->ng_reset_ack);

        ogs_list_for_each(&gnb->ran_ue_list, iter) {
            if (iter->part_of_ng_reset_requested == true) {
                /* The GNB_UE context
                 * where PartOfNG_interface was requested
                 * still remains */
                return;
            }
        }

        /* All GNB_UE context
         * where PartOfNG_interface was requested
         * REMOVED */
        ngap_send_to_gnb(gnb, gnb->ng_reset_ack, NGAP_NON_UE_SIGNALLING);

        /* Clear NG-Reset Ack Buffer */
        gnb->ng_reset_ack = NULL;
        break;
    default:
        ogs_warn("Invalid ResetType[%d]", ResetType->present);
        break;
    }
}
