/*
 * (C) 2013 by Andreas Eversberg <jolly@eversberg.eu>
 * (C) 2016 by Tom Tsou <tom.tsou@ettus.com>
 *
 * All Rights Reserved
 *
 * SPDX-License-Identifier: GPL-2.0+
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <stdint.h>

#include <osmocom/core/bits.h>
#include <osmocom/coding/gsm0503_tables.h>

/*! \addtogroup tables
 *  @{
 *
 *  GSM TS 05.03 tables.
 *
 *  This module contains various tables defining parts of 3GPP TS 05.03
 *  / 45.003, primarily for the purpose of (de)puncturing, interleaving,
 *  etc.
 *
 * \file gsm0503_tables.c */

const ubit_t gsm0503_pdtch_hl_hn_ubit[4][8] = {
	{ 1,1, 1,1, 1,1, 1,1 },
	{ 1,1, 0,0, 1,0, 0,0 },
	{ 0,0, 1,0, 0,0, 0,1 },
	{ 0,0, 0,1, 0,1, 1,0 },
};

const ubit_t gsm0503_pdtch_edge_hl_hn_ubit[3][8] = {
	{ 0,0, 0,1, 0,1, 1,0 },
	{ 0,0, 0,0, 0,0, 0,0 },
	{ 1,1, 1,0, 0,1, 1,1 },
};

const sbit_t gsm0503_pdtch_hl_hn_sbit[4][8] = {
	{ -127,-127, -127,-127, -127,-127, -127,-127 },
	{ -127,-127,  127, 127, -127, 127,  127, 127 },
	{  127, 127, -127, 127,  127, 127,  127,-127 },
	{  127, 127,  127,-127,  127,-127, -127, 127 },
};

const sbit_t gsm0503_pdtch_edge_hl_hn_sbit[3][8] = {
	{  127, 127,  127,-127,  127,-127, -127, 127 },
	{  127, 127,  127, 127,  127, 127,  127, 127 },
	{ -127,-127, -127, 127,  127,-127, -127,-127 },
};

/*
 * 3GPP TS 05.03 sec 5.1.2.2 "Block code". Rows re-ordered to be indxed by USF in host bit order.
 */
const ubit_t gsm0503_usf2six[8][6] = {
	{ 0,0,0, 0,0,0 },
	{ 1,0,0, 1,0,1 },
	{ 0,1,0, 1,1,0 },
	{ 1,1,0, 0,1,1 },
	{ 0,0,1, 0,1,1 },
	{ 1,0,1, 1,1,0 },
	{ 0,1,1, 1,0,1 },
	{ 1,1,1, 0,0,0 },
};

/*
 * 3GPP TS 05.03 sec 5.1.4.2 "Block code". Rows re-ordered to be indxed by USF in host bit order.
 */
const ubit_t gsm0503_usf2twelve_ubit[8][12] = {
	{ 0,0,0, 0,0,0, 0,0,0, 0,0,0 },
	{ 1,1,0, 1,0,0, 0,0,1, 0,1,1 },
	{ 0,0,1, 1,0,1, 1,1,0, 1,1,0 },
	{ 1,1,1, 0,0,1, 1,1,1, 1,0,1 },
	{ 0,0,0, 0,1,1, 0,1,1, 1,0,1 },
	{ 1,1,0, 1,1,1, 0,1,0, 1,1,0 },
	{ 0,0,1, 1,1,0, 1,0,1, 0,1,1 },
	{ 1,1,1, 0,1,0, 1,0,0, 0,0,0 },
};

const sbit_t gsm0503_usf2twelve_sbit[8][12] = {
	{  127, 127, 127,  127, 127, 127,  127, 127, 127,  127, 127, 127 },
	{ -127,-127, 127, -127, 127, 127,  127, 127,-127,  127,-127,-127 },
	{  127, 127,-127, -127, 127,-127, -127,-127, 127, -127,-127, 127 },
	{ -127,-127,-127,  127, 127,-127, -127,-127,-127, -127, 127,-127 },
	{  127, 127, 127,  127,-127,-127,  127,-127,-127, -127, 127,-127 },
	{ -127,-127, 127, -127,-127,-127,  127,-127, 127, -127,-127, 127 },
	{  127, 127,-127, -127,-127, 127, -127, 127,-127,  127,-127,-127 },
	{ -127,-127,-127,  127,-127, 127, -127, 127, 127,  127, 127, 127 },
};

const uint8_t gsm0503_puncture_cs2[588] = {
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,0, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,0, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,0, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,0, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,0, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,0, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,0, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,0, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,0, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,0, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,0, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,1, 0,0,0,0, 0,0,0,1, 0,0,0,1,
	0,0,0,1, 0,0,0,1, 0,0,0,1
};

const uint8_t gsm0503_puncture_cs3[676] = {
	0,0,0,0,0,0, 0,0,0,0,0,0, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,1,0,1,
	0,0,0,1,0,1, 0,0,0,1,0,1, 0,0,0,0
};

const uint8_t gsm0503_puncture_mcs1_dl_hdr[108] = {
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,1,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,1,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,1,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,1,1,
};

const uint8_t gsm0503_puncture_mcs1_ul_hdr[117] = {
	0,0,0,0,0,1,0,0,1,0,0,1,
	0,0,0,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,
};

const uint8_t gsm0503_puncture_mcs1_p1[588] = {
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,0,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
	0,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,0,0,1,
};

const uint8_t gsm0503_puncture_mcs1_p2[588] = {
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,0,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,0,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,0,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,0,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,0,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,0,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,0,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,0,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
	0,1,0,0,1,0,0,1,0,1,0,0,0,1,0,1,1,0,0,1,0,
};

const uint8_t gsm0503_puncture_mcs2_p1[732] = {
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,0,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,0,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,0,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,0,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,0,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,0,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0, 0,0,1,1,1,0,
	0,0,1,1,1,0, 0,0,1,1,1,0,
};

const uint8_t gsm0503_puncture_mcs2_p2[732] = {
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 0,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 0,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 0,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	0,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 0,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 0,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1, 1,1,0,0,0,1,
	1,1,0,0,0,1, 1,1,0,0,0,1,
};

const uint8_t gsm0503_puncture_mcs3_p1[948] = {
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,
	0,0,1,0,1,1,0,1,1,1,0,1,
};

const uint8_t gsm0503_puncture_mcs3_p2[948] = {
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,
};

const uint8_t gsm0503_puncture_mcs3_p3[948] = {
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,
};

const uint8_t gsm0503_puncture_mcs4_p1[1116] = {
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1,
};

const uint8_t gsm0503_puncture_mcs4_p2[1116] = {
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1,
};

const uint8_t gsm0503_puncture_mcs4_p3[1116] = {
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0,
};

const uint8_t gsm0503_puncture_mcs5_p1[1404] = {
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0, 0,0,1,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0, 0,0,1,0,0,1,0,0,1, 0,0,1,0,0,1,0,0,1,
};

const uint8_t gsm0503_puncture_mcs5_p2[1404] = {
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0, 0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0, 0,1,0,0,1,0,0,1,0, 0,1,0,0,1,0,0,1,0,
};

const uint8_t gsm0503_puncture_mcs6_p1[1836] = {
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,0, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,0, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,0, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,0, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,0, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,0, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,0, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,0, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,0, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,0, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1,
	0,0,1, 0,0,1,
};

const uint8_t gsm0503_puncture_mcs6_p2[1836] = {
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,0,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,0,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,0,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,0,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,0,0,
	0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0,
	0,1,0, 0,1,0,
};

const uint8_t gsm0503_puncture_mcs7_dl_hdr[135] = {
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,1,0,0,0,0,0,
	0,0,0,1,0,0,0,0,0,0,
	0,0,0,1,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	1,0,0,0,0,0,0,0,0,1,
	0,0,0,0,0,0,0,0,0,1,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,1,0,0,0,
	0,0,0,0,0,1,0,0,0,0,
	0,0,0,0,0,1,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0,0,
	0,1,0,0,0,
};

const uint8_t gsm0503_puncture_mcs7_ul_hdr[162] = {
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,1,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,1,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,
};

const uint8_t gsm0503_puncture_mcs7_p1[1404] = {
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,0,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
	0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,
};

const uint8_t gsm0503_puncture_mcs7_p2[1404] = {
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,1,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,1,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,1,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,1,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,1,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,1,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,1,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,1,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,1,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,0,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,1,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,1,0,
	1,1,0,0,1,0,0,1,1,1,0,1,1,1,0,1,1,0,
};

const uint8_t gsm0503_puncture_mcs7_p3[1404] = {
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
	1,1,0,1,1,0,0,0,1,0,1,1,0,1,1,1,0,1,
};

const uint8_t gsm0503_puncture_mcs8_p1[1692] = {
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,0,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
	0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,
};

const uint8_t gsm0503_puncture_mcs8_p2[1692] = {
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,0,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
	1,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,1,1,0,1,0,1,1,0,1,1,
};

const uint8_t gsm0503_puncture_mcs8_p3[1692] = {
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,0,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
	1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,
};

const uint8_t gsm0503_puncture_mcs9_p1[1836] = {
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
	0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1,
};

const uint8_t gsm0503_puncture_mcs9_p2[1836] = {
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
	1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
};

const uint8_t gsm0503_puncture_mcs9_p3[1836] = {
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
	1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,
};

const uint16_t gsm0503_interleave_mcs5[1248] = {
	   0,  463,  890, 1038,  220,  371,  795,  946,  582,  733, 1160,   63,  490,  641, 277,  428,
	 852, 1003,  185,  333, 1223,  120,  547,  698, 1122,   28,  915, 1066,  242,  390, 817,  968,
	 610,  761, 1185,   85,  512,  660,  305,  453,  880, 1031,  204,  355,  782, 1242, 148,  575,
	 723, 1150,   50,  474,  625, 1088,  267,  418,  845,  993,  169,  320, 1207,  113, 537,  688,
	1115,   12,  902, 1050,  232,  383,  807,  958,  594,  745, 1172,   75,  502,  653, 289,  440,
	 864, 1015,  197,  345, 1235,  132,  559,  710, 1134,   40,  927, 1078,  254,  402, 829,  980,
	 159,  622,  773, 1197,   97,  524,  672, 1099,    5,  465,  892, 1043,  216,  367, 794,  942,
	 587,  735, 1162,   62,  486,  637,  279,  430,  857, 1005,  181,  332, 1219,  125, 549,  700,
	1127,   24,  914, 1062,  244,  395,  819,  970,  606,  757, 1184,   87,  514,  665, 301,  452,
	 876, 1027,  209,  357,  784, 1247,  144,  571,  722, 1146,   52,  479,  627, 1090, 266,  414,
	 841,  992,  171,  322, 1209,  109,  536,  684, 1111,   17,  904, 1055,  228,  379, 806,  954,
	 599,  747, 1174,   74,  498,  649,  291,  442,  869, 1017,  193,  344, 1231,  137, 561,  712,
	1139,   36,  926, 1074,  256,  407,  831,  982,  158,  618,  769, 1196,   99,  526, 677, 1101,
	   7,  458,  894, 1033,  227,  363,  802,  941,  577,  740, 1152,   70,  485,  645, 284,  420,
	 859,  998,  189,  328, 1215,  127,  542,  702, 1117,   35,  922, 1061,  246,  385, 824,  960,
	 605,  765, 1180,   92,  504,  667,  309,  448,  887, 1023,  211,  350,  786, 1237, 155,  567,
	 730, 1145,   54,  469,  632, 1080,  274,  413,  849,  988,  176,  312, 1202,  117, 532,  695,
	1107,   19,  906, 1045,  239,  375,  814,  953,  589,  752, 1164,   82,  497,  657, 296,  432,
	 871, 1010,  201,  340, 1227,  139,  554,  714, 1129,   47,  934, 1073,  258,  397, 836,  972,
	 166,  617,  777, 1192,  104,  516,  679, 1094,    9,  460,  899, 1035,  223,  362, 798,  937,
	 579,  742, 1157,   66,  481,  644,  286,  425,  861, 1000,  188,  324, 1214,  129, 544,  707,
	1119,   31,  918, 1057,  251,  387,  826,  965,  601,  764, 1176,   94,  509,  669, 308,  444,
	 883, 1022,  213,  352,  791, 1239,  151,  566,  726, 1141,   59,  471,  634, 1085, 270,  409,
	 848,  984,  178,  317, 1204,  116,  528,  691, 1106,   21,  911, 1047,  235,  374, 810,  949,
	 591,  754, 1169,   78,  493,  656,  298,  437,  873, 1012,  200,  336, 1226,  141, 556,  719,
	1131,   43,  930, 1069,  263,  399,  838,  977,  162,  613,  776, 1188,  106,  521, 681, 1096,
	   2,  462,  889, 1040,  219,  370,  797,  945,  584,  732, 1159,   65,  489,  640, 276,  427,
	 854, 1002,  184,  335, 1222,  122,  546,  697, 1124,   27,  917, 1065,  241,  392, 816,  967,
	 609,  760, 1187,   84,  511,  662,  304,  455,  879, 1030,  206,  354,  781, 1244, 147,  574,
	 725, 1149,   49,  476,  624, 1087,  269,  417,  844,  995,  168,  319, 1206,  112, 539,  687,
	1114,   14,  901, 1052,  231,  382,  809,  957,  596,  744, 1171,   77,  501,  652, 288,  439,
	 866, 1014,  196,  347, 1234,  134,  558,  709, 1136,   39,  929, 1077,  253,  404, 828,  979,
	 161,  621,  772, 1199,   96,  523,  674, 1098,    4,  467,  891, 1042,  218,  366, 793,  944,
	 586,  737, 1161,   61,  488,  636,  281,  429,  856, 1007,  180,  331, 1218,  124, 551,  699,
	1126,   26,  913, 1064,  243,  394,  821,  969,  608,  756, 1183,   89,  513,  664, 300,  451,
	 878, 1026,  208,  359,  783, 1246,  146,  570,  721, 1148,   51,  478,  629, 1089, 265,  416,
	 840,  991,  173,  321, 1211,  108,  535,  686, 1110,   16,  903, 1054,  230,  378, 805,  956,
	 598,  749, 1173,   73,  500,  648,  293,  441,  868, 1019,  192,  343, 1230,  136, 563,  711,
	1138,   38,  925, 1076,  255,  406,  833,  981,  157,  620,  768, 1195,  101,  525, 676, 1103,
	   6,  457,  896, 1032,  226,  365,  801,  940,  576,  739, 1154,   69,  484,  647, 283,  422,
	 858,  997,  191,  327, 1217,  126,  541,  704, 1116,   34,  921, 1060,  248,  384, 823,  962,
	 604,  767, 1179,   91,  506,  666,  311,  447,  886, 1025,  210,  349,  788, 1236, 154,  569,
	 729, 1144,   56,  468,  631, 1082,  273,  412,  851,  987,  175,  314, 1201,  119, 531,  694,
	1109,   18,  908, 1044,  238,  377,  813,  952,  588,  751, 1166,   81,  496,  659, 295,  434,
	 870, 1009,  203,  339, 1229,  138,  553,  716, 1128,   46,  933, 1072,  260,  396, 835,  974,
	 165,  616,  779, 1191,  103,  518,  678, 1093,   11,  459,  898, 1037,  222,  361, 800,  936,
	 581,  741, 1156,   68,  480,  643,  285,  424,  863,  999,  187,  326, 1213,  131, 543,  706,
	1121,   30,  920, 1056,  250,  389,  825,  964,  600,  763, 1178,   93,  508,  671, 307,  446,
	 882, 1021,  215,  351,  790, 1241,  150,  565,  728, 1140,   58,  473,  633, 1084, 272,  408,
	 847,  986,  177,  316, 1203,  115,  530,  690, 1105,   23,  910, 1049,  234,  373, 812,  948,
	 593,  753, 1168,   80,  492,  655,  297,  436,  875, 1011,  199,  338, 1225,  143, 555,  718,
	1133,   42,  932, 1068,  262,  401,  837,  976,  164,  612,  775, 1190,  105,  520, 683, 1095,
	   1,  464,  888, 1039,  221,  369,  796,  947,  583,  734, 1158,   64,  491,  639, 278,  426,
	 853, 1004,  183,  334, 1221,  121,  548,  696, 1123,   29,  916, 1067,  240,  391, 818,  966,
	 611,  759, 1186,   86,  510,  661,  303,  454,  881, 1029,  205,  356,  780, 1243, 149,  573,
	 724, 1151,   48,  475,  626, 1086,  268,  419,  843,  994,  170,  318, 1208,  111, 538,  689,
	1113,   13,  900, 1051,  233,  381,  808,  959,  595,  746, 1170,   76,  503,  651, 290,  438,
	 865, 1016,  195,  346, 1233,  133,  560,  708, 1135,   41,  928, 1079,  252,  403, 830,  978,
	 160,  623,  771, 1198,   98,  522,  673, 1100,    3,  466,  893, 1041,  217,  368, 792,  943,
	 585,  736, 1163,   60,  487,  638,  280,  431,  855, 1006,  182,  330, 1220,  123, 550,  701,
	1125,   25,  912, 1063,  245,  393,  820,  971,  607,  758, 1182,   88,  515,  663, 302,  450,
	 877, 1028,  207,  358,  785, 1245,  145,  572,  720, 1147,   53,  477,  628, 1091, 264,  415,
	 842,  990,  172,  323, 1210,  110,  534,  685, 1112,   15,  905, 1053,  229,  380, 804,  955,
	 597,  748, 1175,   72,  499,  650,  292,  443,  867, 1018,  194,  342, 1232,  135, 562,  713,
	1137,   37,  924, 1075,  257,  405,  832,  983,  156,  619,  770, 1194,  100,  527, 675, 1102,
	   8,  456,  895, 1034,  225,  364,  803,  939,  578,  738, 1153,   71,  483,  646, 282,  421,
	 860,  996,  190,  329, 1216,  128,  540,  703, 1118,   33,  923, 1059,  247,  386, 822,  961,
	 603,  766, 1181,   90,  505,  668,  310,  449,  885, 1024,  212,  348,  787, 1238, 153,  568,
	 731, 1143,   55,  470,  630, 1081,  275,  411,  850,  989,  174,  313, 1200,  118, 533,  693,
	1108,   20,  907, 1046,  237,  376,  815,  951,  590,  750, 1165,   83,  495,  658, 294,  433,
	 872, 1008,  202,  341, 1228,  140,  552,  715, 1130,   45,  935, 1071,  259,  398, 834,  973,
	 167,  615,  778, 1193,  102,  517,  680, 1092,   10,  461,  897, 1036,  224,  360, 799,  938,
	 580,  743, 1155,   67,  482,  642,  287,  423,  862, 1001,  186,  325, 1212,  130, 545,  705,
	1120,   32,  919, 1058,  249,  388,  827,  963,  602,  762, 1177,   95,  507,  670, 306,  445,
	 884, 1020,  214,  353,  789, 1240,  152,  564,  727, 1142,   57,  472,  635, 1083, 271,  410,
	 846,  985,  179,  315, 1205,  114,  529,  692, 1104,   22,  909, 1048,  236,  372, 811,  950,
	 592,  755, 1167,   79,  494,  654,  299,  435,  874, 1013,  198,  337, 1224,  142, 557,  717,
	1132,   44,  931, 1070,  261,  400,  839,  975,  163,  614,  774, 1189,  107,  519, 682, 1097,
};

/* this corresponds to the bit-lengths of the individual codec
 * parameters as indicated in Table 1.1 of TS 06.10 */
const uint8_t gsm0503_gsm_fr_map[76] = {
	6, 6, 5, 5, 4, 4, 3, 3, 7, 2, 2, 6, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
	3, 7, 2, 2, 6, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 7, 2, 2, 6, 3, 3,
	3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 7, 2, 2, 6, 3, 3, 3, 3, 3, 3, 3, 3, 3,
	3, 3, 3, 3
};

/* this table describes the 65 most importaint bits from EFR coded
 * bits as indicated in TS 05.03 (3.1.1.1) */
const uint8_t gsm0503_gsm_efr_protected_bits[65] = {
	 39, 40, 41, 42, 43, 44, 48, 87, 45,  2,
	  3,  8, 10, 18, 19, 24, 46, 47,142,143,
	144,145,146,147, 92, 93,195,196, 98,137,
	148, 94,197,149,150, 95,198,  4,  5, 11,
	 12, 16,  9,  6,  7, 13, 17, 20, 96,199,
	  1, 14, 15, 21, 25, 26, 28,151,201,190,
	240, 88,138,191,241
};

/* Encoded in-band data for speech frames */
const ubit_t gsm0503_afs_ic_ubit[4][8] = {
	{ 0,0,0,0,0,0,0,0 },
	{ 0,1,0,1,1,1,0,1 },
	{ 1,0,1,1,1,0,1,0 },
	{ 1,1,1,0,0,1,1,1 },
};

const sbit_t gsm0503_afs_ic_sbit[4][8] = {
	{  127, 127, 127, 127, 127, 127, 127, 127 },
	{  127,-127, 127,-127,-127,-127, 127,-127 },
	{ -127, 127,-127,-127,-127, 127,-127, 127 },
	{ -127,-127,-127, 127, 127,-127,-127,-127 },
};

const ubit_t gsm0503_ahs_ic_ubit[4][4] = {
	{ 0,0,0,0 },
	{ 1,0,0,1 },
	{ 1,1,1,0 },
	{ 0,1,1,1 },
};

const sbit_t gsm0503_ahs_ic_sbit[4][4] = {
	{  127, 127, 127, 127 },
	{ -127, 127, 127,-127 },
	{ -127,-127,-127, 127 },
	{  127,-127,-127,-127 },
};

const uint8_t gsm0503_tch_hr_interleaving[228][2] = {
	{ 0  ,0 }, { 1  ,2 }, { 78 ,1 }, { 79 ,3 }, { 48 ,0 }, { 49 ,2 },
	{ 54 ,1 }, { 55 ,3 }, { 24 ,0 }, { 25 ,2 }, { 30 ,1 }, { 31 ,3 },
	{ 72 ,0 }, { 73 ,2 }, { 6  ,1 }, { 7  ,3 }, { 96 ,0 }, { 97 ,2 },
	{ 12 ,0 }, { 13 ,2 }, { 102,1 }, { 103,3 }, { 60 ,0 }, { 61 ,2 },
	{ 66 ,1 }, { 67 ,3 }, { 90 ,1 }, { 91 ,3 }, { 36 ,0 }, { 37 ,2 },
	{ 42 ,1 }, { 43 ,3 }, { 18 ,1 }, { 19 ,3 }, { 84 ,0 }, { 85 ,2 },
	{ 108,0 }, { 109,2 }, { 2  ,0 }, { 3  ,2 }, { 80 ,1 }, { 81 ,3 },
	{ 50 ,0 }, { 51 ,2 }, { 56 ,1 }, { 57 ,3 }, { 26 ,0 }, { 27 ,2 },
	{ 32 ,1 }, { 33 ,3 }, { 74 ,0 }, { 75 ,2 }, { 8  ,1 }, { 9  ,3 },
	{ 98 ,0 }, { 99 ,2 }, { 14 ,0 }, { 15 ,2 }, { 104,1 }, { 105,3 },
	{ 62 ,0 }, { 63 ,2 }, { 68 ,1 }, { 69 ,3 }, { 92 ,1 }, { 93 ,3 },
	{ 38 ,0 }, { 39 ,2 }, { 44 ,1 }, { 45 ,3 }, { 20 ,1 }, { 21 ,3 },
	{ 86 ,0 }, { 87 ,2 }, { 110,0 }, { 111,2 }, { 4  ,0 }, { 5  ,2 },
	{ 82 ,1 }, { 83 ,3 }, { 52 ,0 }, { 53 ,2 }, { 58 ,1 }, { 59 ,3 },
	{ 28 ,0 }, { 29 ,2 }, { 34 ,1 }, { 35 ,3 }, { 76 ,0 }, { 77 ,2 },
	{ 10 ,1 }, { 11 ,3 }, { 100,0 }, { 101,2 }, { 16 ,0 }, { 17 ,2 },
	{ 106,1 }, { 107,3 }, { 64 ,0 }, { 65 ,2 }, { 70 ,1 }, { 71 ,3 },
	{ 94 ,1 }, { 95 ,3 }, { 40 ,0 }, { 41 ,2 }, { 46 ,1 }, { 47 ,3 },
	{ 22 ,1 }, { 23 ,3 }, { 88 ,0 }, { 89 ,2 }, { 112,0 }, { 113,2 },
	{ 6  ,0 }, { 7  ,2 }, { 84 ,1 }, { 85 ,3 }, { 54 ,0 }, { 55 ,2 },
	{ 60 ,1 }, { 61 ,3 }, { 30 ,0 }, { 31 ,2 }, { 36 ,1 }, { 37 ,3 },
	{ 78 ,0 }, { 79 ,2 }, { 12 ,1 }, { 13 ,3 }, { 102,0 }, { 103,2 },
	{ 18 ,0 }, { 19 ,2 }, { 108,1 }, { 109,3 }, { 66 ,0 }, { 67 ,2 },
	{ 72 ,1 }, { 73 ,3 }, { 96 ,1 }, { 97 ,3 }, { 42 ,0 }, { 43 ,2 },
	{ 48 ,1 }, { 49 ,3 }, { 24 ,1 }, { 25 ,3 }, { 90 ,0 }, { 91 ,2 },
	{ 0  ,1 }, { 1  ,3 }, { 8  ,0 }, { 9  ,2 }, { 86 ,1 }, { 87 ,3 },
	{ 56 ,0 }, { 57 ,2 }, { 62 ,1 }, { 63 ,3 }, { 32 ,0 }, { 33 ,2 },
	{ 38 ,1 }, { 39 ,3 }, { 80 ,0 }, { 81 ,2 }, { 14 ,1 }, { 15 ,3 },
	{ 104,0 }, { 105,2 }, { 20 ,0 }, { 21 ,2 }, { 110,1 }, { 111,3 },
	{ 68 ,0 }, { 69 ,2 }, { 74 ,1 }, { 75 ,3 }, { 98 ,1 }, { 99 ,3 },
	{ 44 ,0 }, { 45 ,2 }, { 50 ,1 }, { 51 ,3 }, { 26 ,1 }, { 27 ,3 },
	{ 92 ,0 }, { 93 ,2 }, { 2  ,1 }, { 3  ,3 }, { 10 ,0 }, { 11 ,2 },
	{ 88 ,1 }, { 89 ,3 }, { 58 ,0 }, { 59 ,2 }, { 64 ,1 }, { 65 ,3 },
	{ 34 ,0 }, { 35 ,2 }, { 40 ,1 }, { 41 ,3 }, { 82 ,0 }, { 83 ,2 },
	{ 16 ,1 }, { 17 ,3 }, { 106,0 }, { 107,2 }, { 22 ,0 }, { 23 ,2 },
	{ 112,1 }, { 113,3 }, { 70 ,0 }, { 71 ,2 }, { 76 ,1 }, { 77 ,3 },
	{ 100,1 }, { 101,3 }, { 46 ,0 }, { 47 ,2 }, { 52 ,1 }, { 53 ,3 },
	{ 28 ,1 }, { 29 ,3 }, { 94 ,0 }, { 95 ,2 }, { 4  ,1 }, { 5  ,3 },
};

/*
 * 3GPP TS 05.03 5.1.9.1.2 "USF precoding"
 */
const ubit_t gsm0503_mcs5_usf_precode_table[8][36] = {
	{ 0,0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0, },
	{ 1,1,1,1,1,0,0,0,0, 1,1,1,1,0,0,0,0,0, 1,1,1,1,1,1,0,0,0, 1,1,1,1,1,0,0,0,1, },
	{ 1,1,1,0,0,1,1,1,0, 1,1,1,0,1,1,1,0,0, 1,1,0,0,0,0,1,1,0, 1,1,0,0,0,1,1,0,0, },
	{ 1,0,0,1,1,1,1,0,0, 1,1,0,0,0,0,0,1,1, 1,0,1,1,1,0,1,1,1, 0,0,1,0,0,1,1,1,1, },
	{ 0,0,0,1,1,0,0,1,1, 0,0,1,0,1,1,0,1,0, 1,0,0,0,0,1,1,0,1, 1,1,1,1,1,1,1,1,0, },
	{ 1,1,0,1,0,1,0,1,1, 0,0,0,1,1,0,1,0,1, 0,1,1,1,0,1,0,1,1, 1,0,0,1,0,1,0,1,1, },
	{ 0,0,1,0,0,1,1,0,1, 1,0,1,1,1,1,1,1,1, 0,1,1,0,1,0,0,0,1, 0,0,1,1,1,0,1,0,0, },
	{ 0,1,1,0,1,0,1,1,1, 0,1,0,1,0,1,1,1,1, 0,0,0,1,1,1,1,1,0, 0,1,0,0,1,0,0,1,1, },
};

/*! @} */
