%% Copyright (C) 2024 by sysmocom - s.f.m.c. GmbH <info@sysmocom.de>
%% Author: Vadim Yanitskiy <vyanitskiy@sysmocom.de>
%%
%% All Rights Reserved
%%
%% SPDX-License-Identifier: AGPL-3.0-or-later
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU Affero General Public License as
%% published by the Free Software Foundation; either version 3 of the
%% License, or (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU Affero General Public License
%% along with this program.  If not, see <https://www.gnu.org/licenses/>.
%%
%% Additional Permission under GNU AGPL version 3 section 7:
%%
%% If you modify this Program, or any covered work, by linking or
%% combining it with runtime libraries of Erlang/OTP as released by
%% Ericsson on https://www.erlang.org (or a modified version of these
%% libraries), containing parts covered by the terms of the Erlang Public
%% License (https://www.erlang.org/EPLICENSE), the licensors of this
%% Program grant you additional permission to convey the resulting work
%% without the need to license the runtime libraries of Erlang/OTP under
%% the GNU Affero General Public License. Corresponding Source for a
%% non-source form of such a combination shall include the source code
%% for the parts of the runtime libraries of Erlang/OTP used as well as
%% that of the covered work.

-module(s1ap_proxy).

-export([handle_pdu/1,
         encode_pdu/1,
         decode_pdu/1]).

-include_lib("kernel/include/logger.hrl").

-include("S1AP-PDU-Descriptions.hrl").
-include("S1AP-PDU-Contents.hrl").
-include("S1AP-Containers.hrl").
-include("S1AP-Constants.hrl").

%% ------------------------------------------------------------------
%% public API
%% ------------------------------------------------------------------

%% Process an S1AP PDU
-spec handle_pdu(binary()) -> binary().
handle_pdu(Data) when is_binary(Data) ->
    case decode_pdu(Data) of
        {ok, Pdu} ->
            ?LOG_INFO("S1AP PDU: ~p", [Pdu]),
            handle_pdu(Data, Pdu);
        {error, {asn1, Error}} ->
            ?LOG_ERROR("S1AP PDU decoding failed: ~p", [Error]),
            Data
    end.


%% ------------------------------------------------------------------
%% private API
%% ------------------------------------------------------------------

%% Encode an S1AP PDU
-spec encode_pdu(tuple()) -> {ok, binary()} |
                             {error, {asn1, tuple()}}.
encode_pdu(Pdu) ->
    'S1AP-PDU-Descriptions':encode('S1AP-PDU', Pdu).


%% Decode an S1AP PDU
-spec decode_pdu(binary()) -> {ok, tuple()} |
                              {error, {asn1, tuple()}}.
decode_pdu(Data) ->
    'S1AP-PDU-Descriptions':decode('S1AP-PDU', Data).


%% Helper function for handle_pdu/2.
%% Attempt to encode a new (modified) S1AP PDU,
%% return a new binary() on success or Data on error.
handle_pdu_new(Data, NewPdu) ->
    case encode_pdu(NewPdu) of
        {ok, NewData} ->
            NewData;
        {error, {asn1, Error}} ->
            ?LOG_ERROR("S1AP PDU encoding failed: ~p", [Error]),
            Data
    end.


%% 9.1.3.1 E-RAB SETUP REQUEST
handle_pdu(Data, {Outcome = initiatingMessage,
                  #'InitiatingMessage'{procedureCode = ?'id-E-RABSetup',
                                       value = Content} = Pdu}) ->
    ?LOG_DEBUG("Patching E-RAB SETUP REQUEST"),
    IEs = handle_ies(Content#'E-RABSetupRequest'.protocolIEs,
                     ?'id-E-RABToBeSetupListBearerSUReq'),
    NewContent = Content#'E-RABSetupRequest'{protocolIEs = IEs},
    handle_pdu_new(Data, {Outcome, Pdu#'InitiatingMessage'{value = NewContent}});

%% 9.1.3.2 E-RAB SETUP RESPONSE
handle_pdu(Data, {Outcome = successfulOutcome,
                  #'SuccessfulOutcome'{procedureCode = ?'id-E-RABSetup',
                                       value = Content} = Pdu}) ->
    ?LOG_DEBUG("Patching E-RAB SETUP RESPONSE"),
    IEs = handle_ies(Content#'E-RABSetupResponse'.protocolIEs,
                     ?'id-E-RABSetupListBearerSURes'),
    NewContent = Content#'E-RABSetupResponse'{protocolIEs = IEs},
    handle_pdu_new(Data, {Outcome, Pdu#'SuccessfulOutcome'{value = NewContent}});

%% TODO: 9.1.3.3 E-RAB MODIFY REQUEST / (Optional) Transport Information

%% 9.1.3.8 E-RAB MODIFICATION INDICATION
handle_pdu(Data, {Outcome = initiatingMessage,
                  #'InitiatingMessage'{procedureCode = ?'id-E-RABModificationIndication',
                                       value = Content} = Pdu}) ->
    ?LOG_DEBUG("Patching E-RAB MODIFICATION INDICATION"),
    IEs = Content#'E-RABModificationIndication'.protocolIEs,
    %% E-RAB to be Modified List
    IEs1 = handle_ies(IEs, ?'id-E-RABToBeModifiedListBearerModInd'),
    %% E-RAB not to be Modified List
    IEs2 = handle_ies(IEs1, ?'id-E-RABNotToBeModifiedListBearerModInd'),
    NewContent = Content#'E-RABModificationIndication'{protocolIEs = IEs2},
    handle_pdu_new(Data, {Outcome, Pdu#'InitiatingMessage'{value = NewContent}});

%% 9.1.4.1 INITIAL CONTEXT SETUP REQUEST
handle_pdu(Data, {Outcome = initiatingMessage,
                  #'InitiatingMessage'{procedureCode = ?'id-InitialContextSetup',
                                       value = Content} = Pdu}) ->
    ?LOG_DEBUG("Patching INITIAL CONTEXT SETUP REQUEST"),
    IEs = handle_ies(Content#'InitialContextSetupRequest'.protocolIEs,
                     ?'id-E-RABToBeSetupListCtxtSUReq'),
    NewContent = Content#'InitialContextSetupRequest'{protocolIEs = IEs},
    handle_pdu_new(Data, {Outcome, Pdu#'InitiatingMessage'{value = NewContent}});

%% 9.1.4.3 INITIAL CONTEXT SETUP RESPONSE
handle_pdu(Data, {Outcome = successfulOutcome,
                  #'SuccessfulOutcome'{procedureCode = ?'id-InitialContextSetup',
                                       value = Content} = Pdu}) ->
    ?LOG_DEBUG("Patching INITIAL CONTEXT SETUP RESPONSE"),
    IEs = handle_ies(Content#'InitialContextSetupResponse'.protocolIEs,
                     ?'id-E-RABSetupListCtxtSURes'),
    NewContent = Content#'InitialContextSetupResponse'{protocolIEs = IEs},
    handle_pdu_new(Data, {Outcome, Pdu#'SuccessfulOutcome'{value = NewContent}});

%% TODO: 9.1.5.2 HANDOVER COMMAND :: (O) UL/DL Transport Layer Address
%% TODO: 9.1.5.4 HANDOVER REQUEST :: (M) Transport Layer Address
%% TODO: 9.1.5.5 HANDOVER REQUEST ACKNOWLEDGE :: (M) Transport Layer Address,
%%                                               (O) UL/DL Transport Layer Address
%% TODO: 9.1.5.8 PATH SWITCH REQUEST :: (M) Transport Layer Address
%% TODO: 9.1.5.9 PATH SWITCH REQUEST ACKNOWLEDGE :: (M) Transport Layer Address

%% Proxy all other messages unmodified
handle_pdu(Data, _Pdu) ->
    Data.


%% Handle a single IE (Information Element)
-spec handle_ie(tuple()) -> tuple().

%% E-RAB SETUP REQUEST related IEs
handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABToBeSetupListBearerSUReq',
                              value = Content}) ->
    %% This IE contains a list of BearerSUReq, so patch inner IEs
    handle_ies(Content, ?'id-E-RABToBeSetupItemBearerSUReq');

handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABToBeSetupItemBearerSUReq',
                              value = Content}) ->
    %% eNB -> MME direction: we pass our MME facing local address
    TLA = transp_layer_addr(mme_loc_addr),
    Content#'E-RABToBeSetupItemBearerSUReq'{transportLayerAddress = TLA};

%% E-RAB SETUP RESPONSE related IEs
handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABSetupListBearerSURes',
                              value = Content}) ->
    %% This IE contains a list of BearerSURes, so patch inner IEs
    handle_ies(Content, ?'id-E-RABSetupItemBearerSURes');

handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABSetupItemBearerSURes',
                              value = Content}) ->
    %% MME -> eNB direction: we pass our eNB facing local address
    TLA = transp_layer_addr(s1gw_bind_addr),
    Content#'E-RABSetupItemBearerSURes'{transportLayerAddress = TLA};

%% E-RAB MODIFICATION INDICATION related IEs
handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABToBeModifiedListBearerModInd',
                              value = Content}) ->
    %% This IE contains a list of BearerModInd, so patch inner IEs
    handle_ies(Content, ?'id-E-RABToBeModifiedItemBearerModInd');

handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABToBeModifiedItemBearerModInd',
                              value = Content}) ->
    %% eNB -> MME direction: we pass our MME facing local address
    TLA = transp_layer_addr(mme_loc_addr),
    Content#'E-RABToBeModifiedItemBearerModInd'{transportLayerAddress = TLA};

handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABNotToBeModifiedListBearerModInd',
                              value = Content}) ->
    %% This IE contains a list of BearerModInd, so patch inner IEs
    handle_ies(Content, ?'id-E-RABNotToBeModifiedItemBearerModInd');

handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABNotToBeModifiedItemBearerModInd',
                              value = Content}) ->
    %% eNB -> MME direction: we pass our MME facing local address
    TLA = transp_layer_addr(mme_loc_addr),
    Content#'E-RABNotToBeModifiedItemBearerModInd'{transportLayerAddress = TLA};

%% INITIAL CONTEXT SETUP REQUEST related IEs
handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABToBeSetupListCtxtSUReq',
                              value = Content}) ->
    %% This IE contains a list of CtxtSUReq, so patch inner IEs
    handle_ies(Content, ?'id-E-RABToBeSetupItemCtxtSUReq');

handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABToBeSetupItemCtxtSUReq',
                              value = Content}) ->
    %% eNB -> MME direction: we pass our MME facing local address
    TLA = transp_layer_addr(mme_loc_addr),
    Content#'E-RABToBeSetupItemCtxtSUReq'{transportLayerAddress = TLA};

%% INITIAL CONTEXT SETUP RESPONSE related IEs
handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABSetupListCtxtSURes',
                              value = Content}) ->
    %% This IE contains a list of CtxtSURes, so patch inner IEs
    handle_ies(Content, ?'id-E-RABSetupItemCtxtSURes');

handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABSetupItemCtxtSURes',
                              value = Content}) ->
    %% MME -> eNB direction: we pass our eNB facing local address
    TLA = transp_layer_addr(s1gw_bind_addr),
    Content#'E-RABSetupItemCtxtSURes'{transportLayerAddress = TLA};

%% Catch-all variant, which should not be called normally
handle_ie(#'ProtocolIE-Field'{value = Content} = IE) ->
    ?LOG_ERROR("[BUG] Unhandled S1AP IE: ~p", [IE]),
    Content.


%% Iterate over the given list of 'ProtocolIE-Field' IEs,
%% calling function handle_ie/1 for IEs matching the given IEI.
-spec handle_ies(list(), integer()) -> list().
handle_ies(IEs, IEI) ->
    handle_ies([], IEs, IEI).

handle_ies(Acc, [IE | IEs], IEI) ->
    case IE of
        #'ProtocolIE-Field'{id = IEI} ->
            NewIE = IE#'ProtocolIE-Field'{value = handle_ie(IE)},
            handle_ies([NewIE | Acc], IEs, IEI);
        _ ->
            handle_ies([IE | Acc], IEs, IEI)
    end;

handle_ies(Acc, [], _) ->
    lists:reverse(Acc).


%% GTP-U IP address (TransportLayerAddress) to be used while patching
-spec transp_layer_addr(atom()) -> binary().
transp_layer_addr(EnvParam) ->
    {ok, AddrStr} = application:get_env(osmo_s1gw, EnvParam),
    {ok, Addr} = inet:parse_address(AddrStr),
    %% sadly, there exists inet:ntoa/1, but not inet:aton/1
    list_to_binary(tuple_to_list(Addr)).

%% vim:set ts=4 sw=4 et:
