%% Copyright (C) 2024 by sysmocom - s.f.m.c. GmbH <info@sysmocom.de>
%% Author: Vadim Yanitskiy <vyanitskiy@sysmocom.de>
%%
%% All Rights Reserved
%%
%% SPDX-License-Identifier: AGPL-3.0-or-later
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU Affero General Public License as
%% published by the Free Software Foundation; either version 3 of the
%% License, or (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU Affero General Public License
%% along with this program.  If not, see <https://www.gnu.org/licenses/>.
%%
%% Additional Permission under GNU AGPL version 3 section 7:
%%
%% If you modify this Program, or any covered work, by linking or
%% combining it with runtime libraries of Erlang/OTP as released by
%% Ericsson on https://www.erlang.org (or a modified version of these
%% libraries), containing parts covered by the terms of the Erlang Public
%% License (https://www.erlang.org/EPLICENSE), the licensors of this
%% Program grant you additional permission to convey the resulting work
%% without the need to license the runtime libraries of Erlang/OTP under
%% the GNU Affero General Public License. Corresponding Source for a
%% non-source form of such a combination shall include the source code
%% for the parts of the runtime libraries of Erlang/OTP used as well as
%% that of the covered work.

-module(s1ap_proxy).

-export([init/0,
         deinit/1,
         handle_pdu/2,
         encode_pdu/1,
         decode_pdu/1]).

-include_lib("kernel/include/logger.hrl").

-include("S1AP-PDU-Descriptions.hrl").
-include("S1AP-PDU-Contents.hrl").
-include("S1AP-Containers.hrl").
-include("S1AP-Constants.hrl").


-record(proxy_state, { }).

-type proxy_state() :: #proxy_state{}.

%% ------------------------------------------------------------------
%% public API
%% ------------------------------------------------------------------

%% Initialize per-connection data
-spec init() -> proxy_state().
init() ->
    #proxy_state{}.


%% De-initialize per-connection data
-spec deinit(proxy_state()) -> ok.
deinit(_S) ->
    ok.


%% Process an S1AP PDU
-spec handle_pdu(binary(), proxy_state()) -> {binary(), proxy_state()}.
handle_pdu(Data, S) when is_binary(Data) ->
    case decode_pdu(Data) of
        {ok, Pdu} ->
            ?LOG_INFO("S1AP PDU: ~p", [Pdu]),
            handle_pdu(Data, Pdu, S);
        {error, {asn1, Error}} ->
            ?LOG_ERROR("S1AP PDU decoding failed: ~p", [Error]),
            {Data, S}
    end.


%% ------------------------------------------------------------------
%% private API
%% ------------------------------------------------------------------

%% Encode an S1AP PDU
-spec encode_pdu(tuple()) -> {ok, binary()} |
                             {error, {asn1, tuple()}}.
encode_pdu(Pdu) ->
    'S1AP-PDU-Descriptions':encode('S1AP-PDU', Pdu).


%% Decode an S1AP PDU
-spec decode_pdu(binary()) -> {ok, tuple()} |
                              {error, {asn1, tuple()}}.
decode_pdu(Data) ->
    'S1AP-PDU-Descriptions':decode('S1AP-PDU', Data).


%% Helper function for handle_pdu/3.
%% Attempt to encode a new (modified) S1AP PDU,
%% return a new binary() on success or Data on error.
handle_pdu_new(Data, NewPdu, S) ->
    case encode_pdu(NewPdu) of
        {ok, NewData} ->
            {NewData, S};
        {error, {asn1, Error}} ->
            ?LOG_ERROR("S1AP PDU encoding failed: ~p", [Error]),
            {Data, S}
    end.


%% 9.1.3.1 E-RAB SETUP REQUEST
handle_pdu(Data, {Outcome = initiatingMessage,
                  #'InitiatingMessage'{procedureCode = ?'id-E-RABSetup',
                                       value = Content} = Pdu}, S0) ->
    ?LOG_DEBUG("Patching E-RAB SETUP REQUEST"),
    {IEs, S1} = handle_ies(Content#'E-RABSetupRequest'.protocolIEs,
                           ?'id-E-RABToBeSetupListBearerSUReq', S0),
    NewContent = Content#'E-RABSetupRequest'{protocolIEs = IEs},
    handle_pdu_new(Data, {Outcome, Pdu#'InitiatingMessage'{value = NewContent}}, S1);

%% 9.1.3.2 E-RAB SETUP RESPONSE
handle_pdu(Data, {Outcome = successfulOutcome,
                  #'SuccessfulOutcome'{procedureCode = ?'id-E-RABSetup',
                                       value = Content} = Pdu}, S0) ->
    ?LOG_DEBUG("Patching E-RAB SETUP RESPONSE"),
    {IEs, S1} = handle_ies(Content#'E-RABSetupResponse'.protocolIEs,
                           ?'id-E-RABSetupListBearerSURes', S0),
    NewContent = Content#'E-RABSetupResponse'{protocolIEs = IEs},
    handle_pdu_new(Data, {Outcome, Pdu#'SuccessfulOutcome'{value = NewContent}}, S1);

%% TODO: 9.1.3.3 E-RAB MODIFY REQUEST / (Optional) Transport Information

%% 9.1.3.8 E-RAB MODIFICATION INDICATION
handle_pdu(Data, {Outcome = initiatingMessage,
                  #'InitiatingMessage'{procedureCode = ?'id-E-RABModificationIndication',
                                       value = Content} = Pdu}, S0) ->
    ?LOG_DEBUG("Patching E-RAB MODIFICATION INDICATION"),
    IEs = Content#'E-RABModificationIndication'.protocolIEs,
    %% E-RAB to be Modified List
    {IEs1, S1} = handle_ies(IEs, ?'id-E-RABToBeModifiedListBearerModInd', S0),
    %% E-RAB not to be Modified List
    {IEs2, S2} = handle_ies(IEs1, ?'id-E-RABNotToBeModifiedListBearerModInd', S1),
    NewContent = Content#'E-RABModificationIndication'{protocolIEs = IEs2},
    handle_pdu_new(Data, {Outcome, Pdu#'InitiatingMessage'{value = NewContent}}, S2);

%% 9.1.4.1 INITIAL CONTEXT SETUP REQUEST
handle_pdu(Data, {Outcome = initiatingMessage,
                  #'InitiatingMessage'{procedureCode = ?'id-InitialContextSetup',
                                       value = Content} = Pdu}, S0) ->
    ?LOG_DEBUG("Patching INITIAL CONTEXT SETUP REQUEST"),
    {IEs, S1} = handle_ies(Content#'InitialContextSetupRequest'.protocolIEs,
                           ?'id-E-RABToBeSetupListCtxtSUReq', S0),
    NewContent = Content#'InitialContextSetupRequest'{protocolIEs = IEs},
    handle_pdu_new(Data, {Outcome, Pdu#'InitiatingMessage'{value = NewContent}}, S1);

%% 9.1.4.3 INITIAL CONTEXT SETUP RESPONSE
handle_pdu(Data, {Outcome = successfulOutcome,
                  #'SuccessfulOutcome'{procedureCode = ?'id-InitialContextSetup',
                                       value = Content} = Pdu}, S0) ->
    ?LOG_DEBUG("Patching INITIAL CONTEXT SETUP RESPONSE"),
    {IEs, S1} = handle_ies(Content#'InitialContextSetupResponse'.protocolIEs,
                           ?'id-E-RABSetupListCtxtSURes', S0),
    NewContent = Content#'InitialContextSetupResponse'{protocolIEs = IEs},
    handle_pdu_new(Data, {Outcome, Pdu#'SuccessfulOutcome'{value = NewContent}}, S1);

%% TODO: 9.1.5.2 HANDOVER COMMAND :: (O) UL/DL Transport Layer Address
%% TODO: 9.1.5.4 HANDOVER REQUEST :: (M) Transport Layer Address
%% TODO: 9.1.5.5 HANDOVER REQUEST ACKNOWLEDGE :: (M) Transport Layer Address,
%%                                               (O) UL/DL Transport Layer Address
%% TODO: 9.1.5.8 PATH SWITCH REQUEST :: (M) Transport Layer Address
%% TODO: 9.1.5.9 PATH SWITCH REQUEST ACKNOWLEDGE :: (M) Transport Layer Address

%% Proxy all other messages unmodified
handle_pdu(Data, _Pdu, S) ->
    {Data, S}.


%% Handle a single IE (Information Element)
-spec handle_ie(tuple(), proxy_state()) -> {tuple(), proxy_state()}.

%% E-RAB SETUP REQUEST related IEs
handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABToBeSetupListBearerSUReq',
                              value = Content}, S) ->
    %% This IE contains a list of BearerSUReq, so patch inner IEs
    handle_ies(Content, ?'id-E-RABToBeSetupItemBearerSUReq', S);

handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABToBeSetupItemBearerSUReq',
                              value = Content}, S) ->
    %% eNB -> MME direction: we pass our MME facing local address
    TLA = transp_layer_addr(mme_loc_addr),
    {Content#'E-RABToBeSetupItemBearerSUReq'{transportLayerAddress = TLA}, S};

%% E-RAB SETUP RESPONSE related IEs
handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABSetupListBearerSURes',
                              value = Content}, S) ->
    %% This IE contains a list of BearerSURes, so patch inner IEs
    handle_ies(Content, ?'id-E-RABSetupItemBearerSURes', S);

handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABSetupItemBearerSURes',
                              value = Content}, S) ->
    %% MME -> eNB direction: we pass our eNB facing local address
    TLA = transp_layer_addr(s1gw_bind_addr),
    {Content#'E-RABSetupItemBearerSURes'{transportLayerAddress = TLA}, S};

%% E-RAB MODIFICATION INDICATION related IEs
handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABToBeModifiedListBearerModInd',
                              value = Content}, S) ->
    %% This IE contains a list of BearerModInd, so patch inner IEs
    handle_ies(Content, ?'id-E-RABToBeModifiedItemBearerModInd', S);

handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABToBeModifiedItemBearerModInd',
                              value = Content}, S) ->
    %% eNB -> MME direction: we pass our MME facing local address
    TLA = transp_layer_addr(mme_loc_addr),
    {Content#'E-RABToBeModifiedItemBearerModInd'{transportLayerAddress = TLA}, S};

handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABNotToBeModifiedListBearerModInd',
                              value = Content}, S) ->
    %% This IE contains a list of BearerModInd, so patch inner IEs
    handle_ies(Content, ?'id-E-RABNotToBeModifiedItemBearerModInd', S);

handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABNotToBeModifiedItemBearerModInd',
                              value = Content}, S) ->
    %% eNB -> MME direction: we pass our MME facing local address
    TLA = transp_layer_addr(mme_loc_addr),
    {Content#'E-RABNotToBeModifiedItemBearerModInd'{transportLayerAddress = TLA}, S};

%% INITIAL CONTEXT SETUP REQUEST related IEs
handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABToBeSetupListCtxtSUReq',
                              value = Content}, S) ->
    %% This IE contains a list of CtxtSUReq, so patch inner IEs
    handle_ies(Content, ?'id-E-RABToBeSetupItemCtxtSUReq', S);

handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABToBeSetupItemCtxtSUReq',
                              value = Content}, S) ->
    %% eNB -> MME direction: we pass our MME facing local address
    TLA = transp_layer_addr(mme_loc_addr),
    {Content#'E-RABToBeSetupItemCtxtSUReq'{transportLayerAddress = TLA}, S};

%% INITIAL CONTEXT SETUP RESPONSE related IEs
handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABSetupListCtxtSURes',
                              value = Content}, S) ->
    %% This IE contains a list of CtxtSURes, so patch inner IEs
    handle_ies(Content, ?'id-E-RABSetupItemCtxtSURes', S);

handle_ie(#'ProtocolIE-Field'{id = ?'id-E-RABSetupItemCtxtSURes',
                              value = Content}, S) ->
    %% MME -> eNB direction: we pass our eNB facing local address
    TLA = transp_layer_addr(s1gw_bind_addr),
    {Content#'E-RABSetupItemCtxtSURes'{transportLayerAddress = TLA}, S};

%% Catch-all variant, which should not be called normally
handle_ie(#'ProtocolIE-Field'{value = Content} = IE, S) ->
    ?LOG_ERROR("[BUG] Unhandled S1AP IE: ~p", [IE]),
    {Content, S}.


%% Iterate over the given list of 'ProtocolIE-Field' IEs,
%% calling function handle_ie/1 for IEs matching the given IEI.
-spec handle_ies(list(), integer(), proxy_state()) -> {list(), proxy_state()}.
handle_ies(IEs, IEI, S) ->
    handle_ies([], IEs, IEI, S).

handle_ies(Acc, [IE | IEs], IEI, S0) ->
    case IE of
        #'ProtocolIE-Field'{id = IEI} ->
            {Content, S1} = handle_ie(IE, S0),
            NewIE = IE#'ProtocolIE-Field'{value = Content},
            handle_ies([NewIE | Acc], IEs, IEI, S1);
        _ ->
            handle_ies([IE | Acc], IEs, IEI, S0)
    end;

handle_ies(Acc, [], _IEI, S) ->
    {lists:reverse(Acc), S}.


%% GTP-U IP address (TransportLayerAddress) to be used while patching
-spec transp_layer_addr(atom()) -> binary().
transp_layer_addr(EnvParam) ->
    {ok, AddrStr} = application:get_env(osmo_s1gw, EnvParam),
    {ok, Addr} = inet:parse_address(AddrStr),
    %% sadly, there exists inet:ntoa/1, but not inet:aton/1
    list_to_binary(tuple_to_list(Addr)).

%% vim:set ts=4 sw=4 et:
