-module(s1ap_proxy_test).

-include_lib("eunit/include/eunit.hrl").
-include("pfcp_mock.hrl").

%% ------------------------------------------------------------------
%% setup functions
%% ------------------------------------------------------------------

-define(TC(Fun), {setup,
                  fun start/0,
                  fun stop/1,
                  Fun}).


start() ->
    pfcp_mock:mock_all(),
    exometer_mock:mock_all(),
    s1ap_proxy:init().


stop(S) ->
    exometer_mock:unmock_all(),
    pfcp_mock:unmock_all(),
    s1ap_proxy:deinit(S).


%% ------------------------------------------------------------------
%% testcase descriptions
%% ------------------------------------------------------------------

s1ap_proxy_test_() ->
    [{"S1 SETUP REQUEST (unmodified)",
      ?TC(fun test_s1_setup_req/1)},
     {"E-RAB SETUP REQUEST/RESPONSE",
      ?TC(fun test_e_rab_setup/1)},
     {"E-RAB SETUP REQUEST (failure)",
      ?TC(fun test_e_rab_setup_req_fail/1)},
     {"E-RAB RELEASE COMMAND/RESPONSE",
      ?TC(fun test_e_rab_release_cmd/1)},
     {"E-RAB RELEASE INDICATION",
      ?TC(fun test_e_rab_release_ind/1)},
     {"E-RAB MODIFICATION INDICATION",
      ?TC(fun test_e_rab_modify_ind/1)},
     {"INITIAL CONTEXT SETUP REQUEST/RESPONSE",
      ?TC(fun test_initial_context_setup/1)}].


%% ------------------------------------------------------------------
%% actual testcases
%% ------------------------------------------------------------------

test_s1_setup_req(S0) ->
    SetupReq = s1_setup_req_pdu(),
    %% Expect the PDU to be proxied unmodified
    [?_assertEqual({{forward, SetupReq}, S0}, s1ap_proxy:process_pdu(SetupReq, S0))].


test_e_rab_setup(S0) ->
    %% [eNB <- MME] E-RAB SETUP REQUEST
    SetupReqIn = e_rab_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    SetupReqExp = e_rab_setup_req_pdu(?ADDR_A2U, ?TEID_A2U),
    {{forward, SetupReqOut}, S1} = s1ap_proxy:process_pdu(SetupReqIn, S0),

    %% [eNB -> MME] E-RAB SETUP RESPONSE
    SetupRspIn = e_rab_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    SetupRspExp = e_rab_setup_rsp_pdu(?ADDR_C2U, ?TEID_C2U),
    {{forward, SetupRspOut}, _S2} = s1ap_proxy:process_pdu(SetupRspIn, S1),

    [?_assertEqual(SetupReqExp, SetupReqOut),
     ?_assertEqual(SetupRspExp, SetupRspOut)].


test_e_rab_setup_req_fail(S0) ->
    %% pfcp_peer:session_establish_req/3 responds with a reject
    PDU = pfcp_mock:pdu_rsp_reject(session_establishment_response, ?SEID_Loc),
    pfcp_mock:mock_req(session_establish_req, PDU),
    %% the linked erab_fsm will terminate abnormally, so trap this
    process_flag(trap_exit, true),
    %% eNB <- [S1GW <- MME] E-RAB SETUP REQUEST
    %% eNB -- [S1GW -> MME] E-RAB SETUP RESPONSE (failure)
    SetupReqIn = e_rab_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    SetupRspExp = e_rab_setup_rsp_fail_pdu(),
    {{reply, SetupRspOut}, _S1} = s1ap_proxy:process_pdu(SetupReqIn, S0),

    %% TODO: make sure that the E-RAB FSM has been terminated

    [?_assertEqual(SetupRspExp, SetupRspOut)].


test_e_rab_release_cmd(S0) ->
    %% [eNB <- MME] E-RAB SETUP REQUEST
    SetupReqIn = e_rab_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    {_, S1} = s1ap_proxy:process_pdu(SetupReqIn, S0),

    %% [eNB -> MME] E-RAB SETUP RESPONSE
    SetupRspIn = e_rab_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    {_, S2} = s1ap_proxy:process_pdu(SetupRspIn, S1),

    %% [eNB <- MME] E-RAB RELEASE COMMAND
    ReleaseCmd = e_rab_release_cmd_pdu(),
    {{forward, ReleaseCmdOut}, S3} = s1ap_proxy:process_pdu(ReleaseCmd, S2),

    %% [eNB -> MME] E-RAB RELEASE RESPONSE
    ReleaseRsp = e_rab_release_rsp_pdu(),
    {{forward, ReleaseRspOut}, _S4} = s1ap_proxy:process_pdu(ReleaseRsp, S3),

    %% TODO: make sure that the E-RAB FSM has been terminated

    [?_assertEqual(ReleaseCmd, ReleaseCmdOut),
     ?_assertEqual(ReleaseRsp, ReleaseRspOut)].


test_e_rab_release_ind(S0) ->
    %% [eNB <- MME] E-RAB SETUP REQUEST
    SetupReqIn = e_rab_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    {_, S1} = s1ap_proxy:process_pdu(SetupReqIn, S0),

    %% [eNB -> MME] E-RAB SETUP RESPONSE
    SetupRspIn = e_rab_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    {_, S2} = s1ap_proxy:process_pdu(SetupRspIn, S1),

    %% [eNB -> MME] E-RAB RELEASE INDICATION
    ReleaseInd = e_rab_release_ind_pdu(),
    {{forward, ReleaseIndOut}, _S3} = s1ap_proxy:process_pdu(ReleaseInd, S2),

    %% TODO: make sure that the E-RAB FSM has been terminated

    [?_assertEqual(ReleaseInd, ReleaseIndOut)].


test_e_rab_modify_ind(S0) ->
    %% [eNB -> MME] E-RAB MODIFICATION INDICATION
    ModifyIndIn = e_rab_modify_ind_pdu(?ADDR_U2A, ?TEID_U2A),
    %% XXX: not implemented, we should actually expect ?ADDR_C2U, ?TEID_C2U
    ModifyIndExp = e_rab_modify_ind_pdu(?ADDR_U2A, ?TEID_U2A),
    {{forward, ModifyIndOut}, _S1} = s1ap_proxy:process_pdu(ModifyIndIn, S0),

    [?_assertEqual(ModifyIndExp, ModifyIndOut)].

test_initial_context_setup(S0) ->
    %% [eNB <- MME] INITIAL CONTEXT SETUP REQUEST
    InitCtxSetupReqIn = initial_context_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    InitCtxSetupReqExp = initial_context_setup_req_pdu(?ADDR_A2U, ?TEID_A2U),
    {{forward, InitCtxSetupReqOut}, S1} = s1ap_proxy:process_pdu(InitCtxSetupReqIn, S0),

    %% [eNB -> MME] INITIAL CONTEXT SETUP RESPONSE
    InitCtxSetupRspIn = initial_context_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    InitCtxSetupRspExp = initial_context_setup_rsp_pdu(?ADDR_C2U, ?TEID_C2U),
    {{forward, InitCtxSetupRspOut}, _S2} = s1ap_proxy:process_pdu(InitCtxSetupRspIn, S1),

    [?_assertEqual(InitCtxSetupReqExp, InitCtxSetupReqOut),
     ?_assertEqual(InitCtxSetupRspExp, InitCtxSetupRspOut)].


%% ------------------------------------------------------------------
%% S1AP PDU templates
%% ------------------------------------------------------------------

%% S1 SETUP REQUEST
s1_setup_req_pdu() ->
    << 16#00, 16#11, 16#00, 16#1a, 16#00, 16#00, 16#02, 16#00,
       16#3b, 16#40, 16#08, 16#00, 16#00, 16#f1, 16#10, 16#00,
       16#00, 16#00, 16#00, 16#00, 16#40, 16#00, 16#07, 16#00,
       16#0c, 16#0e, 16#40, 16#00, 16#f1, 16#10
    >>.


%% [eNB <- MME] E-RAB SETUP REQUEST
e_rab_setup_req_pdu(TLA, TEID) when is_binary(TLA),
                                    is_integer(TEID) ->
    << 16#00, 16#05, 16#00, 16#80, 16#9b, 16#00, 16#00, 16#03,
       16#00, 16#00, 16#00, 16#02, 16#00, 16#07, 16#00, 16#08,
       16#00, 16#02, 16#00, 16#09, 16#00, 16#10, 16#00, 16#80,
       16#87, 16#00, 16#00, 16#11, 16#00, 16#80, 16#81, 16#0c,
       16#00, 16#05, 16#04, 16#0f, 16#80,
       TLA/bytes, %% transportLayerAddress (IPv4)
       TEID:32/big, %% GTP-TEID
       16#72, 16#27, 16#c8,
       16#1a, 16#bc, 16#ec, 16#11, 16#62, 16#54, 16#c1, 16#01,
       16#05, 16#04, 16#03, 16#69, 16#6d, 16#73, 16#05, 16#01,
       16#c0, 16#a8, 16#65, 16#02, 16#5e, 16#02, 16#b3, 16#8c,
       16#58, 16#32, 16#27, 16#54, 16#80, 16#80, 16#21, 16#10,
       16#02, 16#00, 16#00, 16#10, 16#81, 16#06, 16#08, 16#08,
       16#08, 16#08, 16#83, 16#06, 16#08, 16#08, 16#04, 16#04,
       16#00, 16#0d, 16#04, 16#08, 16#08, 16#08, 16#08, 16#00,
       16#0d, 16#04, 16#08, 16#08, 16#04, 16#04, 16#00, 16#03,
       16#10, 16#20, 16#01, 16#48, 16#60, 16#48, 16#60, 16#00,
       16#00, 16#00, 16#00, 16#00, 16#00, 16#00, 16#00, 16#88,
       16#88, 16#00, 16#03, 16#10, 16#20, 16#01, 16#48, 16#60,
       16#48, 16#60, 16#00, 16#00, 16#00, 16#00, 16#00, 16#00,
       16#00, 16#00, 16#88, 16#44, 16#00, 16#0c, 16#04, 16#ac,
       16#16, 16#00, 16#15, 16#00, 16#10, 16#02, 16#05, 16#78
    >>.


%% [eNB -> MME] E-RAB SETUP RESPONSE
e_rab_setup_rsp_pdu(TLA, TEID) when is_binary(TLA),
                                    is_integer(TEID) ->
    << 16#20, 16#05, 16#00, 16#22, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#40, 16#02, 16#00, 16#07, 16#00, 16#08, 16#40,
       16#02, 16#00, 16#09, 16#00, 16#1c, 16#40, 16#0f, 16#00,
       16#00, 16#27, 16#40, 16#0a, 16#0c, 16#1f,
       TLA/bytes, %% transportLayerAddress (IPv4)
       TEID:32/big %% GTP-TEID
    >>.


%% [S1GW -> MME] E-RAB SETUP RESPONSE (failure)
%% Transport-cause: transport-resource-unavailable
e_rab_setup_rsp_fail_pdu() ->
    << 16#20, 16#05, 16#00, 16#1a, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#40, 16#02, 16#00, 16#07, 16#00, 16#08, 16#40,
       16#02, 16#00, 16#09, 16#00, 16#1d, 16#40, 16#07, 16#00,
       16#00, 16#23, 16#40, 16#02, 16#0c, 16#20
    >>.


%% [eNB <- MME] E-RAB RELEASE COMMAND
%% TODO: make E-RAB IDs configurable
e_rab_release_cmd_pdu() ->
    << 16#00, 16#07, 16#00, 16#1a, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#00, 16#02, 16#00, 16#07, 16#00, 16#08, 16#00,
       16#02, 16#00, 16#09, 16#00, 16#21, 16#40, 16#07, 16#00,
       16#00, 16#23, 16#40, 16#02, 16#0c, 16#40
    >>.


%% [eNB -> MME] E-RAB RELEASE RESPONSE
%% TODO: make E-RAB IDs configurable
e_rab_release_rsp_pdu() ->
    << 16#20, 16#07, 16#00, 16#19, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#00, 16#02, 16#00, 16#07, 16#00, 16#08, 16#00,
       16#02, 16#00, 16#09, 16#00, 16#45, 16#40, 16#06, 16#00,
       16#00, 16#0f, 16#40, 16#01, 16#0c
    >>.


%% [eNB -> MME] E-RAB RELEASE INDICATION
%% TODO: make E-RAB IDs configurable
e_rab_release_ind_pdu() ->
    << 16#00, 16#08, 16#00, 16#1a, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#00, 16#02, 16#00, 16#07, 16#00, 16#08, 16#00,
       16#02, 16#00, 16#09, 16#00, 16#6e, 16#40, 16#07, 16#00,
       16#00, 16#23, 16#40, 16#02, 16#0c, 16#40
    >>.


%% [eNB -> MME] E-RAB MODIFICATION INDICATION
e_rab_modify_ind_pdu(TLA, TEID) when is_binary(TLA),
                                     is_integer(TEID) ->
    << 16#00, 16#32, 16#00, 16#24, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#00, 16#02, 16#00, 16#02, 16#00, 16#08, 16#00,
       16#04, 16#80, 16#06, 16#69, 16#2d, 16#00, 16#c7, 16#00,
       16#0f, 16#00, 16#00, 16#c8, 16#00, 16#0a, 16#0a, 16#1f,
       TLA/bytes, %% transportLayerAddress (IPv4)
       TEID:32/big %% GTP-TEID
    >>.


%% [eNB <- MME] INITIAL CONTEXT SETUP REQUEST
initial_context_setup_req_pdu(TLA, TEID) when is_binary(TLA),
                                              is_integer(TEID) ->
    << 16#00, 16#09, 16#00, 16#81, 16#10, 16#00, 16#00, 16#07,
       16#00, 16#00, 16#00, 16#02, 16#00, 16#01, 16#00, 16#08,
       16#00, 16#02, 16#00, 16#01, 16#00, 16#42, 16#00, 16#0a,
       16#18, 16#3b, 16#9a, 16#ca, 16#00, 16#60, 16#3b, 16#9a,
       16#ca, 16#00, 16#00, 16#18, 16#00, 16#80, 16#b5, 16#00,
       16#00, 16#34, 16#00, 16#80, 16#af, 16#45, 16#00, 16#09,
       16#20, 16#0f, 16#80,
       TLA/bytes, %% transportLayerAddress (IPv4)
       TEID:32/big, %% GTP-TEID
       16#80, 16#9f, 16#27, 16#bd, 16#de,
       16#61, 16#4e, 16#02, 16#07, 16#42, 16#02, 16#29, 16#06,
       16#40, 16#32, 16#f8, 16#10, 16#00, 16#01, 16#00, 16#6e,
       16#52, 16#05, 16#c1, 16#01, 16#09, 16#09, 16#08, 16#69,
       16#6e, 16#74, 16#65, 16#72, 16#6e, 16#65, 16#74, 16#05,
       16#01, 16#c0, 16#a8, 16#64, 16#02, 16#5e, 16#06, 16#fe,
       16#fe, 16#fa, 16#fa, 16#02, 16#02, 16#58, 16#32, 16#27,
       16#4d, 16#80, 16#80, 16#21, 16#10, 16#02, 16#00, 16#00,
       16#10, 16#81, 16#06, 16#08, 16#08, 16#08, 16#08, 16#83,
       16#06, 16#08, 16#08, 16#04, 16#04, 16#00, 16#0d, 16#04,
       16#08, 16#08, 16#08, 16#08, 16#00, 16#0d, 16#04, 16#08,
       16#08, 16#04, 16#04, 16#00, 16#03, 16#10, 16#20, 16#01,
       16#48, 16#60, 16#48, 16#60, 16#00, 16#00, 16#00, 16#00,
       16#00, 16#00, 16#00, 16#00, 16#88, 16#88, 16#00, 16#03,
       16#10, 16#20, 16#01, 16#48, 16#60, 16#48, 16#60, 16#00,
       16#00, 16#00, 16#00, 16#00, 16#00, 16#00, 16#00, 16#88,
       16#44, 16#00, 16#10, 16#02, 16#05, 16#aa, 16#50, 16#0b,
       16#f6, 16#32, 16#f8, 16#10, 16#00, 16#02, 16#01, 16#c0,
       16#00, 16#02, 16#86, 16#13, 16#32, 16#f8, 16#10, 16#00,
       16#01, 16#23, 16#05, 16#f4, 16#04, 16#94, 16#25, 16#f8,
       16#64, 16#02, 16#01, 16#08, 16#00, 16#6b, 16#00, 16#05,
       16#1c, 16#00, 16#0e, 16#00, 16#00, 16#00, 16#49, 16#00,
       16#20, 16#57, 16#5d, 16#d3, 16#66, 16#69, 16#73, 16#4c,
       16#8c, 16#48, 16#42, 16#76, 16#5f, 16#fe, 16#f9, 16#bd,
       16#d5, 16#89, 16#08, 16#ae, 16#f6, 16#f4, 16#88, 16#62,
       16#22, 16#d7, 16#80, 16#fb, 16#36, 16#33, 16#d9, 16#9e,
       16#9f, 16#00, 16#c0, 16#40, 16#08, 16#35, 16#87, 16#61,
       16#10, 16#03, 16#ff, 16#ff, 16#74
    >>.


%% [eNB -> MME] INITIAL CONTEXT SETUP RESPONSE
initial_context_setup_rsp_pdu(TLA, TEID) when is_binary(TLA),
                                              is_integer(TEID) ->
    << 16#20, 16#09, 16#00, 16#22, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#40, 16#02, 16#00, 16#01, 16#00, 16#08, 16#40,
       16#02, 16#00, 16#01, 16#00, 16#33, 16#40, 16#0f, 16#00,
       16#00, 16#32, 16#40, 16#0a, 16#0a, 16#1f,
       TLA/bytes, %% transportLayerAddress (IPv4)
       TEID:32/big %% GTP-TEID
    >>.

%% vim:set ts=4 sw=4 et:
