/*
 * Copyright (C) 2019 Andreas Steffen
 * HSR Hochschule fuer Technik Rapperswil
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the Licenseor (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be usefulbut
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include <crypto/crypto_tester.h>

/**
 * NIST SP 800-90A DRBG CTR Validation System (DRBGVS)
 */

/**
 * AES-128 no df test case 1 - count 0
 */
drbg_test_vector_t drbg_ctr_aes128_1 = {
	.type = DRBG_CTR_AES128, .strength = 128,
	{ NULL, 0 },
	chunk_from_chars(0xed, 0x1e, 0x7f, 0x21, 0xef, 0x66, 0xea, 0x5d, 0x8e, 0x2a,
					 0x85, 0xb9, 0x33, 0x72, 0x45, 0x44, 0x5b, 0x71, 0xd6, 0x39,
					 0x3a, 0x4e, 0xec, 0xb0, 0xe6, 0x3c, 0x19, 0x3d, 0x0f, 0x72,
					 0xf9, 0xa9, 0x30, 0x3f, 0xb5, 0x19, 0xf0, 0xa4, 0xe1, 0x7d,
					 0x6d, 0xf0, 0xb6, 0x42, 0x6a, 0xa0, 0xec, 0xb2, 0xa3, 0x60,
					 0x79, 0xbd, 0x48, 0xbe, 0x47, 0xad, 0x2a, 0x8d, 0xbf, 0xe4,
					 0x8d, 0xa3, 0xef, 0xad),
	chunk_from_chars(0xf8, 0x01, 0x11, 0xd0, 0x8e, 0x87, 0x46, 0x72, 0xf3, 0x2f,
					 0x42, 0x99, 0x71, 0x33, 0xa5, 0x21, 0x0f, 0x7a, 0x93, 0x75,
					 0xe2, 0x2c, 0xea, 0x70, 0x58, 0x7f, 0x9c, 0xfa, 0xfe, 0xbe,
					 0x0f, 0x6a, 0x6a, 0xa2, 0xeb, 0x68, 0xe7, 0xdd, 0x91, 0x64,
					 0x53, 0x6d, 0x53, 0xfa, 0x02, 0x0f, 0xca, 0xb2, 0x0f, 0x54,
					 0xca, 0xdd, 0xfa, 0xb7, 0xd6, 0xd9, 0x1e, 0x5f, 0xfe, 0xc1,
					 0xdf, 0xd8, 0xde, 0xaa)
};

/**
 * AES-128 no df test case 3 - count 0
 */
drbg_test_vector_t drbg_ctr_aes128_2 = {
	.type = DRBG_CTR_AES128, .strength = 128,
	chunk_from_chars(0xe8, 0xfa, 0x4c, 0x5d, 0xe8, 0x25, 0x79, 0x1e, 0x68, 0x18,
					 0x0f, 0x2b, 0xa1, 0x07, 0xe8, 0x29, 0xc4, 0x82, 0x99, 0xcb,
					 0x01, 0xbe, 0x93, 0x9c, 0xd0, 0xbe, 0x76, 0xda, 0x12, 0x0a,
					 0x91, 0xf2),
	chunk_from_chars(0x34, 0xcb, 0xc2, 0xb2, 0x17, 0xf3, 0xd9, 0x07, 0xfa, 0x2a,
					 0xd6, 0xa0, 0xd7, 0xa8, 0x13, 0xb0, 0xfd, 0xa1, 0xe1, 0x7f,
					 0xbe, 0xed, 0x94, 0xb0, 0xe0, 0xa0, 0xab, 0xfb, 0xec, 0x94,
					 0x71, 0x46, 0x83, 0x26, 0xf8, 0xe9, 0xcf, 0xbd, 0x02, 0xeb,
					 0x07, 0x6b, 0xbb, 0x98, 0x19, 0xd9, 0x6a, 0x02, 0x38, 0x6f,
					 0x80, 0xbf, 0x91, 0x3c, 0x8e, 0x4a, 0x80, 0x36, 0x1d, 0x82,
					 0xca, 0xfa, 0xd5, 0x2e),
	chunk_from_chars(0x52, 0xf5, 0xe7, 0x18, 0xbf, 0x48, 0xd9, 0x9e, 0x49, 0x87,
					 0x75, 0xc0, 0x03, 0x78, 0xe5, 0x45, 0x79, 0x9b, 0xb2, 0x05,
					 0x9a, 0xef, 0x0b, 0x74, 0xbe, 0x57, 0x3d, 0x82, 0x83, 0xf0,
					 0x2b, 0x52, 0x93, 0x91, 0x79, 0x13, 0xbc, 0x8f, 0x26, 0xfc,
					 0x23, 0x76, 0x0a, 0x1c, 0x86, 0xc3, 0xf5, 0xc8, 0x44, 0x85,
					 0x74, 0x19, 0x86, 0x8e, 0xaf, 0xeb, 0x17, 0xc9, 0x24, 0x82,
					 0x27, 0xd0, 0x26, 0xb8)
};

/**
 * AES-128 no df test case 5 - count 0
 */
drbg_test_vector_t drbg_ctr_aes128_3 = {
	.type = DRBG_CTR_AES128, .strength = 128,
	{ NULL, 0 },
	chunk_from_chars(0xb2, 0xb3, 0x29, 0x83, 0x06, 0x47, 0x1b, 0xfc, 0xae, 0x61,
					 0x43, 0x8a, 0x3a, 0x79, 0xe2, 0x35, 0x5e, 0xfa, 0x0b, 0x6e,
					 0xde, 0x4c, 0xbc, 0xd3, 0xe6, 0x6d, 0xe5, 0x14, 0x0b, 0x3a,
					 0xe6, 0x80, 0x26, 0xf0, 0xd1, 0xe4, 0x4b, 0xe5, 0x75, 0xee,
					 0x6f, 0x3e, 0xda, 0x89, 0xc1, 0xe7, 0xe4, 0xfb, 0xd1, 0x42,
					 0x8f, 0x88, 0x52, 0x60, 0x48, 0x71, 0xc7, 0xa4, 0xf4, 0xc7,
					 0x07, 0xa3, 0x93, 0x28),
	chunk_from_chars(0xb7, 0x1b, 0x6d, 0x73, 0x22, 0x45, 0x3a, 0x61, 0x2c, 0x34,
					 0xa9, 0x1c, 0x1e, 0x5c, 0x3f, 0x8c, 0x30, 0x48, 0x6a, 0x69,
					 0x2b, 0x1a, 0xd1, 0x3a, 0x4c, 0x08, 0xca, 0xcc, 0xd1, 0x23,
					 0xa6, 0x39, 0xfd, 0x2e, 0x0a, 0x7c, 0x38, 0x9c, 0xfa, 0x1a,
					 0x97, 0xcb, 0x78, 0xb4, 0x38, 0xdf, 0xf5, 0x7b, 0x0b, 0x5e,
					 0xc4, 0xd5, 0x69, 0xa8, 0xb2, 0x81, 0x0a, 0x15, 0xf8, 0x5c,
					 0x8c, 0x92, 0x26, 0xbf)
};

/**
 * AES-128 no df test case 7 - count 0
 */
drbg_test_vector_t drbg_ctr_aes128_4 = {
	.type = DRBG_CTR_AES128, .strength = 128,
	chunk_from_chars(0x35, 0x14, 0x20, 0xc0, 0x26, 0x3c, 0xe1, 0x1e, 0xe8, 0xb6,
					 0x83, 0xf6, 0x10, 0x61, 0x30, 0xc6, 0x7f, 0xf1, 0xc6, 0x55,
					 0xc4, 0xe6, 0x78, 0x82, 0x52, 0x93, 0xf0, 0x04, 0xd2, 0x7c,
					 0x54, 0x24),
	chunk_from_chars(0x5c, 0xd1, 0xdf, 0x6d, 0xb5, 0x8e, 0xa5, 0x07, 0x83, 0x8d,
					 0x74, 0x26, 0xb3, 0xfb, 0x48, 0x40, 0x2c, 0xd1, 0x4a, 0xb7,
					 0x5a, 0xbb, 0xde, 0xf3, 0x3c, 0xe3, 0x0f, 0xb9, 0x7c, 0x53,
					 0x09, 0x98, 0x99, 0xe6, 0x85, 0x0f, 0xa2, 0x91, 0x31, 0xbf,
					 0xc7, 0x48, 0xb2, 0xe7, 0x4e, 0x0f, 0xd6, 0x2a, 0xcc, 0x4b,
					 0xe4, 0xe9, 0xb5, 0xf0, 0x64, 0x47, 0xdc, 0x26, 0xf7, 0x72,
					 0xc0, 0x24, 0x15, 0x61),
	chunk_from_chars(0xf6, 0x04, 0x0a, 0xf8, 0xae, 0x7a, 0xb0, 0x4c, 0xde, 0x02,
					 0xbe, 0x25, 0xaf, 0x95, 0xde, 0xda, 0xda, 0x3b, 0x10, 0x32,
					 0x1c, 0x41, 0x8c, 0x7a, 0xf4, 0xed, 0x5b, 0xc8, 0x2e, 0x28,
					 0xeb, 0xf7, 0x78, 0xae, 0x42, 0x48, 0xc5, 0x65, 0x29, 0x2e,
					 0x4c, 0xb8, 0xec, 0xcd, 0x40, 0xf1, 0x8a, 0x38, 0x28, 0x48,
					 0xb4, 0x0d, 0x74, 0x41, 0xa2, 0x91, 0xcc, 0x9e, 0xe8, 0x46,
					 0x5c, 0xbe, 0x5f, 0xd6)
};

/**
 * AES-128 no df test case 9 - count 0
 */
drbg_test_vector_t drbg_ctr_aes128_5 = {
	.type = DRBG_CTR_AES128, .strength = 128,
	{ NULL, 0 },
	chunk_from_chars(0x85, 0x82, 0x04, 0xf2, 0x7e, 0x93, 0xe8, 0x72, 0xa1, 0xce,
					 0x62, 0x08, 0xef, 0x6d, 0xe3, 0x98, 0x68, 0xdd, 0xad, 0x55,
					 0xea, 0xc8, 0x99, 0xca, 0x05, 0x81, 0xb9, 0xbb, 0x69, 0x7d,
					 0x44, 0x17, 0x1e, 0x82, 0x20, 0x08, 0xae, 0xcf, 0x6e, 0x67,
					 0xcc, 0x47, 0xf2, 0xb9, 0x8b, 0x04, 0x68, 0x7b, 0x12, 0x1e,
					 0xa4, 0x03, 0x6b, 0x6c, 0xab, 0xb0, 0xc3, 0x6e, 0xfb, 0xe1,
					 0x51, 0x46, 0x38, 0x0b),
	chunk_from_chars(0x49, 0xce, 0x84, 0x96, 0x3d, 0xf8, 0x64, 0xaa, 0xe3, 0x6b,
					 0xec, 0x3f, 0x3e, 0xd2, 0x23, 0xf9, 0x5d, 0xf4, 0x95, 0x6c,
					 0x7c, 0x01, 0xb6, 0x20, 0xd5, 0xf2, 0x10, 0xe6, 0x38, 0x65,
					 0x9d, 0x5e, 0x18, 0x86, 0x70, 0x0c, 0x04, 0xe7, 0x9e, 0x57,
					 0x9c, 0xa7, 0xc9, 0xb0, 0x54, 0x54, 0xb5, 0x25, 0xc2, 0xea,
					 0x73, 0x48, 0x03, 0x2b, 0x3a, 0xb8, 0xd9, 0x1b, 0x4c, 0x47,
					 0x00, 0x22, 0x04, 0xaf)
};

/**
 * AES-128 no df test case 11 - count 0
 */
drbg_test_vector_t drbg_ctr_aes128_6 = {
	.type = DRBG_CTR_AES128, .strength = 128,
	chunk_from_chars(0xa6, 0xcf, 0xbd, 0xd0, 0x5e, 0xee, 0x36, 0xa3, 0xd1, 0x01,
					 0xc3, 0xd0, 0x30, 0x1c, 0xb6, 0xff, 0x21, 0x68, 0x7d, 0x9f,
					 0x89, 0xcf, 0xc8, 0x60, 0xd0, 0x5d, 0xfa, 0x1a, 0x95, 0x7f,
					 0x56, 0xbd),
	chunk_from_chars(0x87, 0x64, 0xbf, 0x9c, 0x12, 0xff, 0x37, 0xe2, 0x3a, 0xfe,
					 0x28, 0xa4, 0x5c, 0x43, 0x04, 0xc0, 0x5d, 0x54, 0x65, 0x4b,
					 0xa1, 0x4c, 0xf5, 0x8d, 0x7a, 0xe3, 0x71, 0x34, 0x7e, 0x1e,
					 0x70, 0x35, 0x19, 0x09, 0x33, 0xf0, 0x7a, 0x1d, 0x44, 0x8d,
					 0xab, 0x65, 0x07, 0x16, 0xa1, 0xb3, 0x07, 0x14, 0xca, 0xda,
					 0x5c, 0xd0, 0xfb, 0xc4, 0x3b, 0x9d, 0x2d, 0xde, 0x79, 0x1c,
					 0x4a, 0xd8, 0x52, 0x2d),
	chunk_from_chars(0x5c, 0xbd, 0x9b, 0x5c, 0xf1, 0x88, 0x33, 0x30, 0xb3, 0x0b,
					 0x1d, 0xa9, 0x17, 0xe6, 0x2d, 0x66, 0x82, 0x11, 0x5f, 0x0e,
					 0x97, 0x52, 0x39, 0x6c, 0x32, 0xc5, 0x97, 0xb9, 0x0a, 0x95,
					 0x72, 0xa7, 0xe9, 0xc6, 0x13, 0x89, 0xf5, 0xec, 0x97, 0x9a,
					 0x72, 0xf7, 0x1f, 0xe0, 0x3d, 0x0c, 0xd2, 0xf3, 0x4b, 0x3d,
					 0x91, 0x70, 0xcc, 0x96, 0x1a, 0x79, 0x42, 0xc6, 0x52, 0xec,
					 0x42, 0x65, 0x1b, 0xd3)
};

/**
 * AES-192 no df test case 1 - count 0
 */
drbg_test_vector_t drbg_ctr_aes192_1 = {
	.type = DRBG_CTR_AES192, .strength = 192,
	{ NULL, 0 },
	chunk_from_chars(0xd6, 0xe1, 0x8f, 0x45, 0x65, 0xfd, 0xf2, 0x82, 0x6d, 0x0d,
					 0x56, 0x41, 0x96, 0x47, 0xc0, 0x20, 0x41, 0x3b, 0x96, 0x32,
					 0x99, 0xd8, 0xde, 0x2c, 0x65, 0x10, 0x27, 0x7f, 0x8c, 0xe9,
					 0x88, 0xa7, 0xf0, 0xb3, 0xbc, 0x1d, 0xf8, 0x5b, 0x15, 0x3f,
					 0xa8, 0x23, 0xd6, 0x31, 0x1f, 0x9f, 0x66, 0xdf, 0x32, 0x9e,
					 0x3d, 0x70, 0x65, 0xe2, 0x4f, 0xe2, 0x50, 0x7e, 0x6b, 0x9d,
					 0xbc, 0xc2, 0x28, 0x38, 0x48, 0x3f, 0xa7, 0x29, 0xca, 0x51,
					 0x16, 0xd0, 0x3a, 0x91, 0x02, 0x81, 0x39, 0xd7, 0x13, 0x0a),
	chunk_from_chars(0x4b, 0xf8, 0x06, 0x69, 0x0a, 0xf1, 0x3d, 0xbc, 0xfd, 0x44,
					 0x8c, 0x79, 0xa3, 0x53, 0x2e, 0x00, 0x0b, 0xca, 0xbc, 0xef,
					 0x36, 0xf2, 0x64, 0x3f, 0x3e, 0x1c, 0x9d, 0xe6, 0x07, 0x10,
					 0x42, 0x82, 0xf8, 0x1c, 0xd6, 0xcd, 0xcf, 0x8d, 0xa8, 0x42,
					 0x9c, 0x94, 0x10, 0x82, 0x45, 0x11, 0x4d, 0x3d, 0xa1, 0x7b,
					 0x9f, 0x48, 0xbb, 0x07, 0x09, 0x4c, 0x07, 0x3a, 0x94, 0xf5,
					 0xd2, 0xef, 0x9e, 0x30)
};

/**
 * AES-192 no df test case 3 - count 0
 */
drbg_test_vector_t drbg_ctr_aes192_2 = {
	.type = DRBG_CTR_AES192, .strength = 192,
	chunk_from_chars(0x2e, 0xca, 0x97, 0x47, 0x8d, 0x60, 0x6b, 0x0b, 0xca, 0x56,
					 0xf0, 0x55, 0xa1, 0xe8, 0x39, 0x4b, 0x44, 0xf5, 0xe3, 0x02,
					 0x8d, 0xa5, 0x9a, 0x39, 0x00, 0xd5, 0xd7, 0xbd, 0x34, 0x1b,
					 0x06, 0x21, 0xb5, 0xf6, 0xd2, 0x30, 0x9e, 0x36, 0xb4, 0xae),
	chunk_from_chars(0xe0, 0xe0, 0x3d, 0x4a, 0x9e, 0x4d, 0xe3, 0x62, 0xe6, 0xbb,
					 0x9b, 0xa6, 0x35, 0xb8, 0x47, 0x91, 0x2a, 0x3f, 0x9e, 0x1d,
					 0x83, 0xb4, 0x36, 0x3f, 0xc2, 0x58, 0xcd, 0xea, 0x29, 0x28,
					 0xa8, 0x78, 0x23, 0x18, 0x6f, 0x1c, 0x47, 0xc4, 0x17, 0x5f,
					 0x66, 0xaa, 0x3d, 0xb4, 0xbe, 0xa8, 0xf2, 0xe1, 0xb2, 0x46,
					 0x97, 0x51, 0x57, 0x8f, 0x24, 0xf1, 0xb4, 0xd0, 0xdf, 0x97,
					 0x1c, 0xf2, 0x28, 0x17, 0x12, 0x14, 0x98, 0x12, 0xc2, 0x0e,
					 0x80, 0x4b, 0x90, 0xf3, 0xaa, 0xc1, 0x2d, 0x7c, 0x98, 0x32),
	chunk_from_chars(0x0e, 0x02, 0xf3, 0x44, 0xb1, 0x15, 0x9c, 0xcd, 0x77, 0x95,
					 0x97, 0x4e, 0x7b, 0xb9, 0xdf, 0x9e, 0x8c, 0xc3, 0x81, 0xa1,
					 0x30, 0xb9, 0x8b, 0x43, 0x46, 0x22, 0x85, 0xfe, 0x3e, 0xd8,
					 0x58, 0x78, 0x11, 0xab, 0xde, 0xf4, 0x71, 0xbb, 0xcf, 0xe8,
					 0x30, 0xe5, 0x16, 0xb7, 0x8f, 0x78, 0x1a, 0x05, 0xb5, 0x2e,
					 0x44, 0xd5, 0x41, 0x22, 0x77, 0x50, 0x84, 0x17, 0x0f, 0xf6,
					 0xd8, 0x06, 0x27, 0xe9)
};

/**
 * AES-192 no df test case 5 - count 0
 */
drbg_test_vector_t drbg_ctr_aes192_3 = {
	.type = DRBG_CTR_AES192, .strength = 192,
	{ NULL, 0 },
	chunk_from_chars(0x21, 0xa5, 0x3c, 0x17, 0x66, 0x1a, 0xed, 0x92, 0x95, 0x0f,
					 0xf4, 0xfd, 0xf4, 0xab, 0x8f, 0xa5, 0xdc, 0x44, 0xe9, 0x9f,
					 0x88, 0xaa, 0x95, 0x3e, 0x19, 0xb4, 0xbe, 0x78, 0xba, 0x75,
					 0xf2, 0x67, 0xf6, 0x70, 0x34, 0xa1, 0x56, 0x6e, 0x83, 0x3e,
					 0x1e, 0xcb, 0x61, 0xa5, 0xdd, 0x70, 0xb1, 0x67, 0xd6, 0x55,
					 0x77, 0x41, 0x5c, 0x2f, 0xb8, 0x12, 0xea, 0x20, 0xb3, 0x25,
					 0xd2, 0xb5, 0x31, 0x58, 0x69, 0x7c, 0xf9, 0x7c, 0x81, 0xd7,
					 0x9e, 0x84, 0x99, 0xf7, 0x45, 0x40, 0x13, 0x7f, 0xd1, 0xe1),
	chunk_from_chars(0x11, 0xa0, 0x7b, 0x22, 0x31, 0x91, 0x6f, 0x99, 0xc3, 0xf7,
					 0xa0, 0x7d, 0x14, 0x51, 0xab, 0x94, 0x0d, 0x2d, 0x66, 0x3c,
					 0x97, 0x29, 0xc1, 0x85, 0x71, 0x70, 0xa3, 0x16, 0x55, 0x07,
					 0x8a, 0x82, 0x74, 0xec, 0x4a, 0x90, 0x31, 0xbc, 0xfd, 0x2a,
					 0xbb, 0x1a, 0xe5, 0x81, 0x3f, 0x49, 0x70, 0xa7, 0x16, 0xe3,
					 0xd1, 0x42, 0x87, 0xe4, 0xf2, 0x1e, 0x01, 0x32, 0x3f, 0xdf,
					 0xff, 0xb7, 0x61, 0x87)
};

/**
 * AES-192 no df test case 7 - count 0
 */
drbg_test_vector_t drbg_ctr_aes192_4 = {
	.type = DRBG_CTR_AES192, .strength = 192,
	chunk_from_chars(0x08, 0x94, 0x30, 0xcf, 0x99, 0x6e, 0x06, 0xa0, 0xdc, 0x31,
					 0x32, 0xc8, 0xaa, 0x6e, 0x3e, 0xdc, 0x76, 0x51, 0xff, 0x4f,
					 0x01, 0xe7, 0x11, 0x3c, 0xa3, 0xce, 0x2a, 0x21, 0x5e, 0x10,
					 0xec, 0x88, 0xad, 0x72, 0x36, 0xb8, 0x30, 0x7c, 0x85, 0x69),
	chunk_from_chars(0x9d, 0xda, 0x17, 0x5d, 0x9f, 0x26, 0xe5, 0x6b, 0x94, 0x49,
					 0xbb, 0x0c, 0x9d, 0x02, 0x2c, 0x47, 0x13, 0x8a, 0xe7, 0x8b,
					 0xa3, 0x8c, 0x35, 0xd1, 0x5f, 0x2d, 0x2c, 0x55, 0x5d, 0x69,
					 0xd4, 0xd6, 0x94, 0x5a, 0x57, 0xae, 0x39, 0x69, 0x62, 0x52,
					 0xd7, 0x5a, 0xae, 0x1b, 0x7b, 0xb4, 0x9b, 0x81, 0xff, 0x43,
					 0xa8, 0xe8, 0x9f, 0x7b, 0x0c, 0xcb, 0xf6, 0xf6, 0xaf, 0x46,
					 0x47, 0xa5, 0x57, 0xf7, 0x6e, 0xd7, 0x3f, 0x09, 0x25, 0x4d,
					 0x67, 0xff, 0xd7, 0xe3, 0x56, 0x29, 0x86, 0x86, 0x23, 0x00),
	chunk_from_chars(0xbc, 0x80, 0x62, 0x76, 0x71, 0xbc, 0xa2, 0x67, 0xba, 0xe1,
					 0xa0, 0xde, 0x9e, 0xb9, 0xe4, 0x81, 0xd8, 0x3b, 0x5c, 0xad,
					 0xe6, 0xe3, 0xc5, 0x08, 0x24, 0x34, 0xb0, 0xb7, 0x00, 0xf3,
					 0x15, 0x43, 0x9c, 0x2c, 0xa2, 0xe9, 0xde, 0xf4, 0x3b, 0x1b,
					 0xf2, 0x05, 0x84, 0x57, 0x7c, 0xef, 0x5b, 0x61, 0xc0, 0x68,
					 0x1e, 0x4b, 0xd4, 0x40, 0xe8, 0x8d, 0xa3, 0x79, 0xa0, 0x21,
					 0x4c, 0xf5, 0xf5, 0xcf)
};

/**
 * AES-192 no df test case 9 - count 0
 */
drbg_test_vector_t drbg_ctr_aes192_5 = {
	.type = DRBG_CTR_AES192, .strength = 192,
	{ NULL, 0 },
	chunk_from_chars(0x3b, 0xb9, 0xb2, 0x66, 0x2a, 0x25, 0x5d, 0xf8, 0xf8, 0x80,
					 0x88, 0x47, 0x18, 0x84, 0x06, 0x96, 0x0e, 0xc0, 0x03, 0x35,
					 0x1c, 0xab, 0x3a, 0xd2, 0x02, 0x78, 0x71, 0xf1, 0x30, 0x29,
					 0x56, 0xaa, 0x31, 0xab, 0xc8, 0xd3, 0x9c, 0x36, 0x29, 0xd9,
					 0x7b, 0xc8, 0x63, 0xda, 0x2b, 0x9f, 0xde, 0x00, 0x8b, 0x61,
					 0x00, 0x79, 0xf8, 0xbd, 0x80, 0x41, 0x80, 0xa7, 0xe8, 0x65,
					 0x3c, 0xa6, 0xe9, 0x1a, 0x5a, 0x90, 0x61, 0x9c, 0x23, 0xe0,
					 0x2c, 0xe4, 0xd6, 0x07, 0xb0, 0x0c, 0x79, 0x24, 0x36, 0xfe),
	chunk_from_chars(0x50, 0x6c, 0x74, 0x51, 0xe5, 0xee, 0x0f, 0xdc, 0xa4, 0x86,
					 0x32, 0x94, 0x2f, 0x2b, 0xa4, 0x73, 0x04, 0xeb, 0x02, 0x10,
					 0xf1, 0xfa, 0xc6, 0x6c, 0x62, 0x03, 0x65, 0xf7, 0x53, 0xef,
					 0x70, 0x98, 0x9c, 0x40, 0xfe, 0x9f, 0xd5, 0x4d, 0x54, 0xa9,
					 0xe4, 0xaa, 0x0b, 0x3f, 0x1a, 0xb7, 0x00, 0xaa, 0x6e, 0x9d,
					 0x45, 0xbc, 0x7c, 0xe3, 0x2b, 0x88, 0xbb, 0xed, 0x53, 0x7e,
					 0xcb, 0xc4, 0xf3, 0x77)
};

/**
 * AES-192 no df test case 11 - count 0
 */
drbg_test_vector_t drbg_ctr_aes192_6 = {
	.type = DRBG_CTR_AES192, .strength = 192,
	chunk_from_chars(0x7a, 0xa6, 0x0c, 0x5a, 0x25, 0x36, 0x0d, 0x8a, 0x0a, 0x9d,
					 0x41, 0xe3, 0x4c, 0x76, 0xc4, 0xf1, 0x80, 0x71, 0xb5, 0x6a,
					 0xc6, 0x73, 0xd4, 0xba, 0xac, 0xbe, 0xe7, 0x75, 0x49, 0xdd,
					 0x36, 0x1d, 0xb6, 0x49, 0x6f, 0x90, 0x34, 0x51, 0x38, 0x88),
	chunk_from_chars(0xe7, 0xb5, 0xcc, 0x99, 0xdd, 0xe3, 0x12, 0x31, 0x8e, 0xcb,
					 0x2a, 0x3b, 0xc4, 0xee, 0x8c, 0x14, 0x76, 0x34, 0x5a, 0xd3,
					 0xe8, 0x1b, 0x5d, 0x6d, 0xc0, 0xb7, 0x47, 0xb5, 0x90, 0xc0,
					 0xd2, 0x1b, 0xe7, 0x31, 0x39, 0x8f, 0x82, 0x3a, 0xad, 0x02,
					 0x31, 0x0e, 0x8e, 0x8d, 0xbc, 0x6a, 0xe5, 0x86, 0x62, 0x5d,
					 0x75, 0x0f, 0x5a, 0x1c, 0xee, 0x76, 0x4a, 0x7e, 0x47, 0xa2,
					 0xeb, 0x35, 0x5c, 0x5c, 0x94, 0xdd, 0xe9, 0x50, 0x06, 0x5f,
					 0xb7, 0x55, 0x46, 0xff, 0x34, 0x7f, 0x7f, 0x17, 0x24, 0xeb),
	chunk_from_chars(0x21, 0x46, 0x0b, 0xda, 0x25, 0xdc, 0xad, 0x96, 0xd2, 0x91,
					 0x44, 0xc9, 0x12, 0xc6, 0x54, 0xe3, 0x07, 0xa0, 0xb5, 0x59,
					 0xd7, 0x26, 0xc0, 0x51, 0x3e, 0x2f, 0x07, 0xa4, 0xe6, 0xca,
					 0x0c, 0x44, 0x4e, 0xcb, 0x4f, 0xf6, 0xa9, 0x77, 0x88, 0x0c,
					 0xba, 0x4b, 0xb9, 0x93, 0xb2, 0xd2, 0x87, 0x3d, 0xab, 0xd3,
					 0x59, 0xa8, 0xc4, 0x93, 0xdc, 0x28, 0x38, 0xab, 0xb1, 0x0f,
					 0x63, 0xd3, 0xa8, 0xe7)
};

/**
 * AES-256 no df test case 1 - count 0
 */
drbg_test_vector_t drbg_ctr_aes256_1 = {
	.type = DRBG_CTR_AES256, .strength = 256,
	{ NULL, 0 },
	chunk_from_chars(0xe4, 0xbc, 0x23, 0xc5, 0x08, 0x9a, 0x19, 0xd8, 0x6f, 0x41,
					 0x19, 0xcb, 0x3f, 0xa0, 0x8c, 0x0a, 0x49, 0x91, 0xe0, 0xa1,
					 0xde, 0xf1, 0x7e, 0x10, 0x1e, 0x4c, 0x14, 0xd9, 0xc3, 0x23,
					 0x46, 0x0a, 0x7c, 0x2f, 0xb5, 0x8e, 0x0b, 0x08, 0x6c, 0x6c,
					 0x57, 0xb5, 0x5f, 0x56, 0xca, 0xe2, 0x5b, 0xad, 0xfd, 0x85,
					 0xa8, 0x36, 0xbb, 0xa8, 0x50, 0x19, 0x88, 0x1e, 0x8c, 0x6b,
					 0xad, 0x23, 0xc9, 0x06, 0x1a, 0xdc, 0x75, 0x47, 0x76, 0x59,
					 0xac, 0xae, 0xa8, 0xe4, 0xa0, 0x1d, 0xfe, 0x07, 0xa1, 0x83,
					 0x2d, 0xad, 0x1c, 0x13, 0x6f, 0x59, 0xd7, 0x0f, 0x86, 0x53,
					 0xa5, 0xdc, 0x11, 0x86, 0x63, 0xd6),
	chunk_from_chars(0xb2, 0xcb, 0x89, 0x05, 0xc0, 0x5e, 0x59, 0x50, 0xca, 0x31,
					 0x89, 0x50, 0x96, 0xbe, 0x29, 0xea, 0x3d, 0x5a, 0x3b, 0x82,
					 0xb2, 0x69, 0x49, 0x55, 0x54, 0xeb, 0x80, 0xfe, 0x07, 0xde,
					 0x43, 0xe1, 0x93, 0xb9, 0xe7, 0xc3, 0xec, 0xe7, 0x3b, 0x80,
					 0xe0, 0x62, 0xb1, 0xc1, 0xf6, 0x82, 0x02, 0xfb, 0xb1, 0xc5,
					 0x2a, 0x04, 0x0e, 0xa2, 0x47, 0x88, 0x64, 0x29, 0x52, 0x82,
					 0x23, 0x4a, 0xaa, 0xda)
};

/**
 * AES-256 no df test case 3 - count 0
 */
drbg_test_vector_t drbg_ctr_aes256_2 = {
	.type = DRBG_CTR_AES256, .strength = 256,
	chunk_from_chars(0x1d, 0x2b, 0xe6, 0xf2, 0x5e, 0x88, 0xfa, 0x30, 0xc4, 0xef,
					 0x42, 0xe4, 0xd5, 0x4e, 0xfd, 0x95, 0x7d, 0xec, 0x23, 0x1f,
					 0xa0, 0x01, 0x43, 0xca, 0x47, 0x58, 0x0b, 0xe6, 0x66, 0xa8,
					 0xc1, 0x43, 0xa9, 0x16, 0xc9, 0x0b, 0x38, 0x19, 0xa0, 0xa7,
					 0xea, 0x91, 0x4e, 0x3c, 0x9a, 0x2e, 0x7a, 0x3f),
	chunk_from_chars(0xff, 0xad, 0x10, 0x10, 0x00, 0x25, 0xa8, 0x79, 0x67, 0x2f,
					 0xf5, 0x03, 0x74, 0xb2, 0x86, 0x71, 0x2f, 0x45, 0x7d, 0xd0,
					 0x14, 0x41, 0xd7, 0x6a, 0xc1, 0xa1, 0xcd, 0x15, 0xc7, 0x39,
					 0x0d, 0xd9, 0x31, 0x79, 0xa2, 0xf5, 0x92, 0x0d, 0x19, 0x8b,
					 0xf3, 0x4a, 0x1b, 0x76, 0xfb, 0xc2, 0x12, 0x89, 0x6c, 0x1a,
					 0x08, 0x9c, 0xae, 0x31, 0x33, 0x63, 0xbc, 0x76, 0xa7, 0x80,
					 0x13, 0x9e, 0xb4, 0xf2, 0xf2, 0x04, 0x8b, 0x1f, 0x6b, 0x07,
					 0x89, 0x6c, 0x5c, 0x41, 0x2b, 0xff, 0x03, 0x85, 0x44, 0x0f,
					 0xc4, 0x3b, 0x73, 0xfa, 0xcb, 0xb7, 0x9e, 0x3a, 0x25, 0x2f,
					 0xa0, 0x1f, 0xe1, 0x7a, 0xb3, 0x91),
	chunk_from_chars(0xe0, 0x53, 0xc7, 0xd4, 0xbd, 0x90, 0x99, 0xef, 0x6a, 0x99,
					 0xf1, 0x90, 0xa5, 0xfd, 0x80, 0x21, 0x94, 0x37, 0xd6, 0x42,
					 0x00, 0x66, 0x72, 0x33, 0x8d, 0xa6, 0xe0, 0xfe, 0x73, 0xca,
					 0x4d, 0x24, 0xff, 0xa5, 0x11, 0x51, 0xbf, 0xbd, 0xac, 0x78,
					 0xd8, 0xa2, 0xf6, 0x25, 0x50, 0x46, 0xed, 0xf5, 0x7a, 0x04,
					 0x62, 0x6e, 0x99, 0x77, 0x13, 0x9c, 0x69, 0x33, 0x27, 0x42,
					 0x99, 0xf3, 0xbd, 0xff)
};

/**
 * AES-256 no df test case 5 - count 0
 */
drbg_test_vector_t drbg_ctr_aes256_3 = {
	.type = DRBG_CTR_AES256, .strength = 256,
	{ NULL, 0 },
	chunk_from_chars(0xfc, 0xa0, 0xb6, 0xe5, 0x57, 0xf5, 0x14, 0x30, 0xdd, 0x78,
					 0x7a, 0xb4, 0xd3, 0x3f, 0x18, 0xd9, 0xae, 0xc2, 0x19, 0x62,
					 0x9d, 0x1b, 0x83, 0x9a, 0x35, 0xca, 0xaf, 0xc8, 0x25, 0xab,
					 0x55, 0xbe, 0x6a, 0x88, 0x03, 0x21, 0xbe, 0x58, 0xc1, 0x6e,
					 0xac, 0xb9, 0x45, 0xb7, 0xbb, 0x7c, 0xad, 0xb7, 0x7b, 0x68,
					 0x49, 0x23, 0xae, 0x50, 0x86, 0x6f, 0x71, 0x0d, 0x3b, 0x5b,
					 0x2e, 0xdf, 0x24, 0x45, 0x59, 0x3f, 0xe6, 0x6d, 0x15, 0xf2,
					 0xdc, 0x73, 0x5e, 0x2b, 0x0c, 0x27, 0x8f, 0x1c, 0xc9, 0x73,
					 0x50, 0x75, 0xdd, 0x26, 0x8b, 0x91, 0x40, 0x8e, 0xf7, 0x3d,
					 0x55, 0x04, 0x23, 0x54, 0x5a, 0xdc),
	chunk_from_chars(0x29, 0x37, 0x56, 0x71, 0x40, 0x7b, 0x1a, 0x45, 0xb9, 0xfd,
					 0x8b, 0x4f, 0x50, 0x7d, 0xda, 0x23, 0x48, 0x22, 0xd5, 0x57,
					 0xe4, 0x8b, 0x9a, 0x52, 0x99, 0x7b, 0x13, 0x75, 0x7e, 0x0c,
					 0x2f, 0x79, 0x6c, 0x87, 0x41, 0xf9, 0x4b, 0xc2, 0xbd, 0x0d,
					 0x8c, 0x98, 0xe5, 0xe2, 0x5e, 0x4c, 0xee, 0x65, 0xe4, 0xdd,
					 0x63, 0x4c, 0x98, 0xbc, 0x3b, 0x66, 0xfb, 0x62, 0x79, 0xf9,
					 0x46, 0x8a, 0xc8, 0x41)
};

/**
 * AES-256 no df test case 7 - count 0
 */
drbg_test_vector_t drbg_ctr_aes256_4 = {
	.type = DRBG_CTR_AES256, .strength = 256,
	chunk_from_chars(0x71, 0x00, 0xbe, 0xe1, 0xf8, 0xca, 0x38, 0xf4, 0xf0, 0x7b,
					 0x99, 0x10, 0xb1, 0x2b, 0xae, 0xce, 0x71, 0x52, 0x22, 0x66,
					 0x3a, 0x1d, 0x5c, 0x16, 0x99, 0xb5, 0xd4, 0x02, 0x2c, 0x0e,
					 0x0b, 0x1a, 0x49, 0xc9, 0x4b, 0x89, 0x8e, 0x53, 0x18, 0xf6,
					 0x86, 0x1b, 0x43, 0xa8, 0xf1, 0xa4, 0xa8, 0x82),
	chunk_from_chars(0x91, 0xe1, 0x5b, 0x57, 0x88, 0x61, 0x56, 0xd0, 0xea, 0xe2,
					 0xed, 0xa3, 0x68, 0x7c, 0xc4, 0xb6, 0x17, 0x72, 0x56, 0x47,
					 0xfc, 0x34, 0x23, 0xfb, 0x54, 0x8f, 0x18, 0x03, 0x38, 0x06,
					 0x4a, 0xb6, 0x68, 0x98, 0x00, 0x50, 0x09, 0xc2, 0xc9, 0xc5,
					 0xf7, 0xc4, 0x20, 0xd9, 0x9e, 0x4f, 0x35, 0x1a, 0x25, 0xda,
					 0x97, 0x00, 0xe7, 0x98, 0x8a, 0x46, 0xb2, 0xfb, 0x44, 0x35,
					 0x8f, 0xc3, 0xb1, 0x40, 0xaf, 0x96, 0xb9, 0xf8, 0x5c, 0xfc,
					 0x74, 0x79, 0x78, 0xe8, 0x5a, 0xfc, 0xca, 0x0b, 0xcc, 0x02,
					 0xe8, 0x07, 0xaf, 0x83, 0x0b, 0x3c, 0x0e, 0x69, 0x60, 0xa6,
					 0x0b, 0xbc, 0x2d, 0xed, 0x89, 0x1b),
	chunk_from_chars(0xea, 0xa8, 0x0c, 0x6f, 0x59, 0x0f, 0x28, 0xcc, 0x7b, 0x5e,
					 0xdb, 0xd3, 0xd8, 0x64, 0x3a, 0x68, 0xf7, 0xe6, 0xde, 0x87,
					 0x3b, 0x0b, 0x9d, 0x83, 0x9b, 0x0a, 0xb9, 0x6c, 0xa2, 0x48,
					 0xbb, 0x4b, 0x92, 0x34, 0xb1, 0xc0, 0x65, 0x85, 0x7d, 0x93,
					 0x6c, 0xe6, 0xdd, 0x0f, 0xc9, 0x2d, 0x6b, 0x3c, 0xf9, 0x8f,
					 0x3a, 0x29, 0xc1, 0x6b, 0xb5, 0x49, 0xf6, 0xde, 0xa4, 0x22,
					 0x12, 0x26, 0xe5, 0x50)
};

/**
 * AES-256 no df test case 9 - count 0
 */
drbg_test_vector_t drbg_ctr_aes256_5 = {
	.type = DRBG_CTR_AES256, .strength = 256,
	{ NULL, 0 },
	chunk_from_chars(0x3f, 0xa9, 0x04, 0x74, 0x70, 0x34, 0xcc, 0x30, 0x93, 0xde,
					 0xaa, 0xc4, 0x97, 0xe3, 0xc1, 0x43, 0xfa, 0x44, 0x00, 0xac,
					 0xcf, 0xc5, 0x58, 0x85, 0x71, 0x7f, 0xa9, 0x43, 0xf4, 0x3c,
					 0xba, 0xd1, 0xa8, 0x91, 0x68, 0xaa, 0x76, 0x96, 0x1e, 0x15,
					 0x0e, 0x26, 0x49, 0xec, 0x1e, 0xd6, 0x73, 0x61, 0xc6, 0x0c,
					 0x5b, 0x41, 0x5b, 0xad, 0x71, 0x54, 0x93, 0x48, 0x6b, 0x7a,
					 0x12, 0x3b, 0xa6, 0xc0, 0x46, 0x08, 0x9e, 0x95, 0x49, 0xea,
					 0x8b, 0xb2, 0x2a, 0x7a, 0xd4, 0x10, 0x8b, 0xec, 0x98, 0x11,
					 0x7f, 0x75, 0x1a, 0x2e, 0x4c, 0xc2, 0x0b, 0x02, 0x51, 0x0d,
					 0x2a, 0x3d, 0x02, 0x60, 0x5b, 0x4d),
	chunk_from_chars(0xab, 0x2d, 0x70, 0x9d, 0xe8, 0x81, 0x16, 0x4b, 0x6c, 0x21,
					 0x49, 0xb2, 0x1e, 0xae, 0x15, 0x17, 0xf8, 0x75, 0x61, 0x64,
					 0x9e, 0x0d, 0xd9, 0xca, 0x5c, 0xe5, 0x51, 0xc5, 0xbd, 0x12,
					 0xfd, 0xf7, 0x09, 0x1e, 0x08, 0x1d, 0x30, 0x71, 0x23, 0xae,
					 0xc5, 0xae, 0x7c, 0x30, 0xaf, 0xd2, 0xa6, 0x4c, 0x8d, 0x13,
					 0x6e, 0xa0, 0x7f, 0x7e, 0xc5, 0xed, 0xb4, 0x40, 0x0b, 0x9a,
					 0x64, 0x45, 0x66, 0x42)
};

/**
 * AES-256 no df test case 11 - count 0
 */
drbg_test_vector_t drbg_ctr_aes256_6 = {
	.type = DRBG_CTR_AES256, .strength = 256,
	chunk_from_chars(0xda, 0x7c, 0x74, 0x2b, 0x40, 0x8d, 0xeb, 0x1b, 0x02, 0x6e,
					 0xc5, 0xdf, 0xeb, 0x00, 0xdd, 0x07, 0x5f, 0x48, 0x06, 0x9c,
					 0x18, 0x5e, 0x5d, 0x35, 0x5b, 0x09, 0xef, 0xf8, 0x8f, 0xcc,
					 0xf2, 0x89, 0xef, 0x04, 0x52, 0x26, 0xc2, 0xe2, 0x99, 0x1e,
					 0x20, 0xb0, 0x97, 0x64, 0x33, 0x99, 0x4c, 0x0d),
	chunk_from_chars(0xa3, 0x68, 0x73, 0x75, 0x12, 0x9c, 0x98, 0x86, 0xea, 0x48,
					 0xa2, 0xf4, 0x9b, 0xe3, 0x28, 0xdd, 0x2b, 0xcf, 0x46, 0x68,
					 0x9a, 0x59, 0xde, 0x69, 0xa9, 0x29, 0xdc, 0xb0, 0x1e, 0x6b,
					 0x79, 0xac, 0x96, 0xf9, 0x8d, 0xde, 0xd9, 0xe1, 0x38, 0x11,
					 0xc2, 0x5c, 0x55, 0x59, 0x7b, 0xbd, 0x3f, 0x8b, 0x40, 0xca,
					 0x11, 0x4f, 0x31, 0xa5, 0x45, 0xb9, 0x29, 0xc4, 0x22, 0x5d,
					 0x0d, 0x21, 0x99, 0x74, 0x3a, 0x5d, 0xf3, 0x6a, 0x83, 0x61,
					 0x89, 0x2d, 0x5c, 0xdf, 0x35, 0x21, 0x8e, 0xed, 0x63, 0x54,
					 0xa6, 0x5c, 0xaf, 0x04, 0xd8, 0x61, 0xf6, 0x14, 0x75, 0x62,
					 0x5b, 0x21, 0x5a, 0xc6, 0x38, 0x3c),
	chunk_from_chars(0x47, 0xa4, 0x52, 0x1d, 0xd4, 0x5c, 0x7b, 0x72, 0xe3, 0x9b,
					 0x7b, 0xb6, 0xbe, 0x14, 0xbf, 0xb4, 0x02, 0x9f, 0x33, 0xea,
					 0x87, 0xbf, 0x11, 0xf1, 0x84, 0x1e, 0x01, 0xd3, 0x37, 0x2d,
					 0x7a, 0x0c, 0x6d, 0x12, 0x89, 0xe0, 0xec, 0x60, 0x59, 0x9c,
					 0x28, 0xc4, 0x0f, 0x38, 0x2f, 0x7d, 0xce, 0x33, 0xcf, 0x81,
					 0x57, 0x55, 0x20, 0xbf, 0xf5, 0x58, 0x00, 0x87, 0xf3, 0x01,
					 0x08, 0x80, 0xbd, 0xbc)
};