/*
 * Copyright (C) 2016 Andreas Steffen
 * HSR Hochschule fuer Technik Rapperswil
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the Licenseor (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be usefulbut
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include <crypto/crypto_tester.h>

/**
 * SHAKE-128 vectors from "https://github.com/gvanas/KeccakCodePackage/"
 */
xof_test_vector_t shake_128_0 = {
	.alg = XOF_SHAKE_128, .len = 0,
	.seed	= "",
	.out_len = 512,
	.out	= "\x7F\x9C\x2B\xA4\xE8\x8F\x82\x7D\x61\x60\x45\x50\x76\x05\x85\x3E"
			  "\xD7\x3B\x80\x93\xF6\xEF\xBC\x88\xEB\x1A\x6E\xAC\xFA\x66\xEF\x26"
			  "\x3C\xB1\xEE\xA9\x88\x00\x4B\x93\x10\x3C\xFB\x0A\xEE\xFD\x2A\x68"
			  "\x6E\x01\xFA\x4A\x58\xE8\xA3\x63\x9C\xA8\xA1\xE3\xF9\xAE\x57\xE2"
			  "\x35\xB8\xCC\x87\x3C\x23\xDC\x62\xB8\xD2\x60\x16\x9A\xFA\x2F\x75"
			  "\xAB\x91\x6A\x58\xD9\x74\x91\x88\x35\xD2\x5E\x6A\x43\x50\x85\xB2"
			  "\xBA\xDF\xD6\xDF\xAA\xC3\x59\xA5\xEF\xBB\x7B\xCC\x4B\x59\xD5\x38"
			  "\xDF\x9A\x04\x30\x2E\x10\xC8\xBC\x1C\xBF\x1A\x0B\x3A\x51\x20\xEA"
			  "\x17\xCD\xA7\xCF\xAD\x76\x5F\x56\x23\x47\x4D\x36\x8C\xCC\xA8\xAF"
			  "\x00\x07\xCD\x9F\x5E\x4C\x84\x9F\x16\x7A\x58\x0B\x14\xAA\xBD\xEF"
			  "\xAE\xE7\xEE\xF4\x7C\xB0\xFC\xA9\x76\x7B\xE1\xFD\xA6\x94\x19\xDF"
			  "\xB9\x27\xE9\xDF\x07\x34\x8B\x19\x66\x91\xAB\xAE\xB5\x80\xB3\x2D"
			  "\xEF\x58\x53\x8B\x8D\x23\xF8\x77\x32\xEA\x63\xB0\x2B\x4F\xA0\xF4"
			  "\x87\x33\x60\xE2\x84\x19\x28\xCD\x60\xDD\x4C\xEE\x8C\xC0\xD4\xC9"
			  "\x22\xA9\x61\x88\xD0\x32\x67\x5C\x8A\xC8\x50\x93\x3C\x7A\xFF\x15"
			  "\x33\xB9\x4C\x83\x4A\xDB\xB6\x9C\x61\x15\xBA\xD4\x69\x2D\x86\x19"
			  "\xF9\x0B\x0C\xDF\x8A\x7B\x9C\x26\x40\x29\xAC\x18\x5B\x70\xB8\x3F"
			  "\x28\x01\xF2\xF4\xB3\xF7\x0C\x59\x3E\xA3\xAE\xEB\x61\x3A\x7F\x1B"
			  "\x1D\xE3\x3F\xD7\x50\x81\xF5\x92\x30\x5F\x2E\x45\x26\xED\xC0\x96"
			  "\x31\xB1\x09\x58\xF4\x64\xD8\x89\xF3\x1B\xA0\x10\x25\x0F\xDA\x7F"
			  "\x13\x68\xEC\x29\x67\xFC\x84\xEF\x2A\xE9\xAF\xF2\x68\xE0\xB1\x70"
			  "\x0A\xFF\xC6\x82\x0B\x52\x3A\x3D\x91\x71\x35\xF2\xDF\xF2\xEE\x06"
			  "\xBF\xE7\x2B\x31\x24\x72\x1D\x4A\x26\xC0\x4E\x53\xA7\x5E\x30\xE7"
			  "\x3A\x7A\x9C\x4A\x95\xD9\x1C\x55\xD4\x95\xE9\xF5\x1D\xD0\xB5\xE9"
			  "\xD8\x3C\x6D\x5E\x8C\xE8\x03\xAA\x62\xB8\xD6\x54\xDB\x53\xD0\x9B"
			  "\x8D\xCF\xF2\x73\xCD\xFE\xB5\x73\xFA\xD8\xBC\xD4\x55\x78\xBE\xC2"
			  "\xE7\x70\xD0\x1E\xFD\xE8\x6E\x72\x1A\x3F\x7C\x6C\xCE\x27\x5D\xAB"
			  "\xE6\xE2\x14\x3F\x1A\xF1\x8D\xA7\xEF\xDD\xC4\xC7\xB7\x0B\x5E\x34"
			  "\x5D\xB9\x3C\xC9\x36\xBE\xA3\x23\x49\x1C\xCB\x38\xA3\x88\xF5\x46"
			  "\xA9\xFF\x00\xDD\x4E\x13\x00\xB9\xB2\x15\x3D\x20\x41\xD2\x05\xB4"
			  "\x43\xE4\x1B\x45\xA6\x53\xF2\xA5\xC4\x49\x2C\x1A\xDD\x54\x45\x12"
			  "\xDD\xA2\x52\x98\x33\x46\x2B\x71\xA4\x1A\x45\xBE\x97\x29\x0B\x6F"
};

xof_test_vector_t shake_128_32 = {
	.alg = XOF_SHAKE_128, .len = 32,
	.seed	= "\x9F\x2F\xCC\x7C\x90\xDE\x09\x0D\x6B\x87\xCD\x7E\x97\x18\xC1\xEA"
			  "\x6C\xB2\x11\x18\xFC\x2D\x5D\xE9\xF9\x7E\x5D\xB6\xAC\x1E\x9C\x10",
	.out_len = 512,
	.out	= "\xFC\xDE\xAD\x82\xF3\x9C\xDF\xCE\xF9\x9C\x1B\xAB\xB6\x74\xA9\xA8"
			  "\xE2\x4A\xC5\x94\x64\x6C\x31\xD0\x20\xA4\xCD\x2B\xC2\x55\x4A\xFD"
			  "\x78\xC4\xE4\x13\xF7\x80\x4F\xA1\x70\x8B\x9F\x40\x00\xFD\x86\x0E"
			  "\x30\x70\xF2\xE1\xBA\x9E\xEE\x38\x05\x35\x2A\xAD\x65\x5B\x4B\x0A"
			  "\x72\x8F\x2D\x5F\xCC\x43\x24\x38\x25\xBC\x0D\xCE\x33\xCA\x71\x66"
			  "\x26\xDC\x76\xE9\x20\xD7\x25\x75\xE2\x6D\xDD\x71\x10\xD0\xF9\x91"
			  "\xA9\x12\x00\xB5\x13\xAE\xE2\x3A\xC9\xBC\x70\x43\xA1\x52\xAC\xE0"
			  "\xCD\x0B\x49\x18\x1D\x2B\xB6\xBD\x36\xE9\x3C\x0B\x62\x7A\xCA\x9C"
			  "\x6A\xB6\xC8\x5E\xD7\x0C\xE7\x62\x42\x9C\x8F\x26\x27\x08\x10\x32"
			  "\x84\xC0\xA7\x92\x13\x8F\x10\xE8\x56\x8E\xFB\x23\x99\xB3\x8A\x31"
			  "\x05\x5C\x11\x88\xBA\x59\x34\x4E\x6A\x2B\x73\xD5\xC0\x4A\xA5\x24"
			  "\x05\x66\x49\x84\x4D\x1D\xAD\xCD\x07\xD3\x5D\xF5\xD8\x51\xEB\xAF"
			  "\xFC\xA5\x70\x3B\x80\x15\x3E\xA6\x27\xB1\xBA\xDF\xB2\x88\x5F\x70"
			  "\xF7\x86\xD3\x4F\x56\x50\xFE\x73\xE3\x69\x0A\x8A\x96\x61\x00\x59"
			  "\x25\x3D\xD3\xAB\xB5\xFA\x7C\x54\xCF\x6E\x77\x69\x5D\x24\xA6\x59"
			  "\x40\x77\xEE\x4D\x36\x73\xF9\xFC\x56\xC6\x2F\xC7\xF7\x10\xCF\x87"
			  "\x20\x14\xC0\xA7\xDE\x8B\x1C\xA6\xAE\x8C\xEF\xAA\xDE\xAF\x5F\x4D"
			  "\x05\x5F\xF7\x65\xAD\x41\x87\x13\xF2\xDD\x08\xEA\xFB\x5E\x16\xEE"
			  "\xD9\xFE\x34\x4E\xE8\xD4\x38\x8F\xDC\x22\x35\x1F\x63\x83\x40\x17"
			  "\xB5\x39\xE3\xFF\x14\x3F\x39\x4B\x5B\x74\xD0\x6F\x65\xE9\x6A\x7A"
			  "\x3D\x02\x8F\xD1\x4F\x6C\x70\x01\xEB\x7A\xD2\xDC\xFC\xF4\xB2\x44"
			  "\x7A\xA1\x73\xA2\xAE\x8E\xDB\x58\x1B\x5B\xBD\x89\xE8\xA4\x68\xFE"
			  "\x0A\x38\x50\x75\x30\xB9\x79\x5D\xA3\xBC\xEC\x6D\xDE\xBC\xE9\xEB"
			  "\x31\x32\xEF\x18\xC9\xC2\xA8\xB9\x36\xA4\x31\xC8\xB1\x21\xFA\x99"
			  "\x6F\xF9\xBA\x5C\xE5\x22\x98\x6B\x67\x8A\x5E\xC9\x9A\x10\x3A\x91"
			  "\xCF\x33\x19\x6E\x08\xC8\x2D\xC6\x5E\x68\xAE\xD2\x38\xA9\x31\x6A"
			  "\x73\xE7\x1C\xF5\xA6\x7C\xE4\x40\xB9\x3B\xDB\x84\x5B\x3A\x60\x53"
			  "\x9E\xCC\xDC\xE4\x1B\xC7\x23\xEC\x9A\x14\xEE\x4E\x08\x2F\x60\xC0"
			  "\xBE\x3D\x5E\x50\xDF\xC8\xBE\x1E\x86\xA9\x7E\xCE\xE9\xD8\x8E\x2B"
			  "\xB2\xA3\xAF\xF4\x7F\xBD\x6D\x66\x75\xD8\x1E\xFE\x07\x08\x92\x6B"
			  "\x81\xAB\x31\x4A\x52\x4F\xC7\x45\x09\x01\x62\xD2\xAC\x72\x3C\x43"
			  "\x26\xE0\xF9\xE1\x6F\xBD\xBA\x2B\x1E\x99\x14\xBB\xEE\xDF\xF9\x6B"
};

xof_test_vector_t shake_128_167 = {
	.alg = XOF_SHAKE_128, .len = 167,
	.seed	= "\x0D\x8D\x09\xAE\xD1\x9F\x10\x13\x96\x9C\xE5\xE7\xEB\x92\xF8\x3A"
			  "\x20\x9A\xE7\x6B\xE3\x1C\x75\x48\x44\xEA\x91\x16\xCE\xB3\x9A\x22"
			  "\xEB\xB6\x00\x30\x17\xBB\xCF\x26\x55\x5F\xA6\x62\x41\x85\x18\x7D"
			  "\xB8\xF0\xCB\x35\x64\xB8\xB1\xC0\x6B\xF6\x85\xD4\x7F\x32\x86\xED"
			  "\xA2\x0B\x83\x35\x8F\x59\x9D\x20\x44\xBB\xF0\x58\x3F\xAB\x8D\x78"
			  "\xF8\x54\xFE\x0A\x59\x61\x83\x23\x0C\x5E\xF8\xE5\x44\x26\x75\x0E"
			  "\xAF\x2C\xC4\xE2\x9D\x3B\xDD\x03\x7E\x73\x4D\x86\x3C\x2B\xD9\x78"
			  "\x9B\x4C\x24\x30\x96\x13\x8F\x76\x72\xC2\x32\x31\x4E\xFF\xDF\xC6"
			  "\x51\x34\x27\xE2\xDA\x76\x91\x6B\x52\x48\x93\x3B\xE3\x12\xEB\x5D"
			  "\xDE\x4C\xF7\x08\x04\xFB\x25\x8A\xC5\xFB\x82\xD5\x8D\x08\x17\x7A"
			  "\xC6\xF4\x75\x60\x17\xFF\xF5",
	.out_len = 512,
	.out	= "\xC7\x3D\x8F\xAA\xB5\xD0\xB4\xD6\x60\xBD\x50\x82\xE4\x4C\x3C\xAC"
			  "\x97\xE6\x16\x48\xBE\x0A\x04\xB1\x16\x72\x4E\x6F\x6B\x65\x76\x84"
			  "\x67\x4B\x4B\x0E\x90\xD0\xAE\x96\xC0\x85\x3E\xBD\x83\x7B\xD8\x24"
			  "\x9A\xDB\xD3\xB6\x0A\x1A\xD1\xFC\xF8\xA6\xAB\x8E\x2F\x5A\xA7\xFF"
			  "\x19\x7A\x3D\x7D\xBE\xDE\xFB\x43\x3B\x61\x35\x36\xAE\xC4\xD6\x55"
			  "\xB7\xBC\xD7\x78\x52\x6B\xE6\x67\x84\x7A\xCD\x2E\x05\x64\xD9\x6C"
			  "\xE5\x14\x0C\x91\x35\x7F\xAD\xE0\x00\xEF\xCB\x40\x45\x7E\x1B\x6C"
			  "\xED\x41\xFA\x10\x2E\x36\xE7\x99\x79\x2D\xB0\x3E\x9A\x40\xC7\x99"
			  "\xBC\xA9\x12\x62\x94\x8E\x17\x60\x50\x65\xFB\xF6\x38\xFB\x40\xA1"
			  "\x57\xB4\x5C\xF7\x91\x1A\x75\x3D\x0D\x20\x5D\xF8\x47\x16\xA5\x71"
			  "\x12\xBE\xAB\x44\xF6\x20\x1F\xF7\x5A\xAD\xE0\xBA\xFB\xA5\x04\x74"
			  "\x5C\xFE\x23\xE4\xE6\x0E\x67\xE3\x99\x36\x22\xAE\xD7\x3A\x1D\xD6"
			  "\xA4\x65\xBD\x45\x3D\xD3\xC5\xBA\x7D\x2C\xDF\x3F\x1D\x39\x37\x6A"
			  "\x67\xC2\x3E\x55\x5F\x5A\xCF\x25\xBC\xE1\xE5\x5F\x30\x72\x52\xB9"
			  "\xAA\xC2\xC0\xA3\x9C\x88\x5C\x7E\x44\xF2\x04\xCB\x82\x1C\x0D\x37"
			  "\xA2\x2D\xE3\xA7\x1F\x3A\x19\x09\xB1\x1B\x71\x81\xC4\x2B\xE9\xB7"
			  "\x8A\xA0\xD0\xA1\x54\x44\xF3\x30\x00\x75\x54\xBC\xFC\xC0\xD8\xFD"
			  "\x87\xD6\x43\x1F\xB9\x3C\x7C\xC3\x87\x67\xA5\x5D\x30\xD3\x54\x55"
			  "\x60\xBD\x38\x0D\xB8\xC4\xC0\xED\xA9\x39\x9F\x68\xF8\x54\x64\x42"
			  "\x66\xC1\xB7\x95\x8B\x27\x0E\x75\xB7\x91\x34\xAA\x01\xE7\xDC\xF1"
			  "\xE6\xFD\xB6\xD9\xAE\x5D\x02\xCC\xE8\xCE\x8E\x48\x04\x75\xE9\x61"
			  "\x7C\xC4\x2A\x91\xC0\x8D\x9A\xF6\xE5\x10\x1B\x8A\xC5\x83\x4A\xDB"
			  "\x2C\x66\x98\x7F\x42\xA5\x80\xBB\x50\x3A\x4B\x34\xA9\xF1\x5A\xDC"
			  "\xD0\xE2\x3D\x0D\x40\x29\x47\x97\x64\x83\x1D\x06\xB5\xCA\xF3\xF1"
			  "\x4B\x91\x44\x9F\x15\xA2\x91\xF4\xAC\x25\x0B\x27\x0B\x6C\xB3\xC3"
			  "\x04\x72\x5C\x99\xE3\x26\x45\xE1\xFC\x02\xA0\xCD\xDD\x9E\x71\x79"
			  "\x11\xF2\x34\x2D\x94\x82\xF8\xE0\x97\x99\x85\xA0\x17\x0D\x72\x5D"
			  "\xAB\x4E\xA6\x6D\x44\xF6\x26\xBA\x47\x59\x25\xFA\x39\xFC\x9D\xEE"
			  "\x92\x9C\x06\xD0\x09\x41\x6C\x0A\xDC\x1D\x98\x7C\xD6\x25\xA2\x0A"
			  "\xCB\xA4\xCC\x87\xF7\x2F\x61\x08\x67\xC3\xA7\xA9\x28\xA3\xA0\x37"
			  "\x96\x76\xE8\xFE\x25\x71\x07\xAB\x2F\x5C\x03\x0B\xD2\x48\x0E\x3D"
			  "\x18\x63\x56\x2E\x1F\xD0\x79\x02\x80\x33\x3E\xD9\xD5\xDD\x5A\x5C"
};

xof_test_vector_t shake_128_168 = {
	.alg = XOF_SHAKE_128, .len = 168,
	.seed	= "\xC3\x23\x6B\x73\xDE\xB7\x66\x2B\xF3\xF3\xDA\xA5\x8F\x13\x7B\x35"
			  "\x8B\xA6\x10\x56\x0E\xF7\x45\x57\x85\xA9\xBE\xFD\xB0\x35\xA0\x66"
			  "\xE9\x07\x04\xF9\x29\xBD\x96\x89\xCE\xF0\xCE\x3B\xDA\x5A\xCF\x44"
			  "\x80\xBC\xEB\x8D\x09\xD1\x0B\x09\x8A\xD8\x50\x0D\x9B\x60\x71\xDF"
			  "\xC3\xA1\x4A\xF6\xC7\x75\x11\xD8\x1E\x3A\xA8\x84\x49\x86\xC3\xBE"
			  "\xA6\xF4\x69\xF9\xE0\x21\x94\xC9\x28\x68\xCD\x5F\x51\x64\x62\x56"
			  "\x79\x8F\xF0\x42\x49\x54\xC1\x43\x4B\xDF\xED\x9F\xAC\xB3\x90\xB0"
			  "\x7D\x34\x2E\x99\x29\x36\xE0\xF8\x8B\xFD\x0E\x88\x4A\x0D\xDB\x67"
			  "\x9D\x05\x47\xCC\xDE\xC6\x38\x42\x85\xA4\x54\x29\xD1\x15\xAC\x7D"
			  "\x23\x5A\x71\x72\x42\x02\x1D\x1D\xC3\x56\x41\xF5\xF0\xA4\x8E\x84"
			  "\x45\xDB\xA5\x8E\x6C\xB2\xC8\xEA",
	.out_len = 512,
	.out	= "\x4A\x05\xF2\xEF\x1A\xAD\x5F\xF4\x30\x64\x29\xEC\x0F\x19\x04\x40"
			  "\x77\xFB\x64\xBF\xE1\xDC\xC5\x0F\x74\xC3\xF0\x45\xE9\xA9\xC3\xDE"
			  "\x4A\x3B\x59\x63\xAE\xF7\x71\xB0\x49\x11\x1B\x7B\x46\x40\xE2\x0B"
			  "\x1B\xA8\x4E\xD7\xAF\xEE\x32\x55\x71\xAC\xF3\x47\xE3\x11\xF3\x3C"
			  "\x1D\x42\x1F\x21\xD6\x63\x06\x5C\x4D\xAD\xDB\xD1\x78\x5C\x5D\xAC"
			  "\x0D\x55\x4C\xED\xB1\xA4\x5A\x32\xE2\x81\x45\xE9\x8F\x49\xDE\xE2"
			  "\x85\xB3\x3D\xE1\x4C\x33\x6D\x10\x95\x0E\xCC\x30\x96\x6B\x79\xE8"
			  "\x61\x3F\xFE\xBB\x70\x2F\xCC\x00\xA1\xC4\x25\x0D\xD3\x85\xAB\xB5"
			  "\x37\xA2\x84\xE9\x10\x8D\x16\xB6\xF0\x8F\x4E\x10\x3F\x2C\x5E\x9E"
			  "\x5C\x87\x9C\xB5\x09\x55\x34\x15\x1E\x3C\x9A\x31\x6D\x06\xDC\xE5"
			  "\x3B\x7F\x01\xB4\x24\xD3\x75\xB5\x64\xFE\x68\x39\xD1\xD1\xF0\x0A"
			  "\x2E\x62\x60\x40\x60\xA9\x74\x8B\xCD\xC8\x14\x37\x37\x95\x9F\xAB"
			  "\xBC\xAE\x18\x51\x21\x3E\x6D\xC2\x8B\xEF\xDA\x48\x14\x9D\xE6\xAA"
			  "\xF4\xA6\x0D\x46\x15\xBE\xD6\x7D\x11\x79\x6F\x61\x73\xC3\xDC\xF1"
			  "\x39\x03\x7B\x31\xEE\xC9\xA8\x40\x4D\xF0\x75\x97\xBC\x26\x6D\x3C"
			  "\x7D\x9E\xB9\xA7\xCA\xBF\x74\x9F\xB4\x4E\x40\xD7\x46\xD0\xE9\xDF"
			  "\xB5\xC8\xBB\xEB\x25\xE3\xF1\x61\x2D\x03\xD3\xEB\x0C\x15\x4D\xE4"
			  "\xB2\x70\x8C\x4F\x8A\x89\x76\x2E\x17\x1F\x74\x45\x18\xAE\xC1\x34"
			  "\xA0\x2E\xEA\xF4\x9D\xB2\xE2\xC6\xC9\x91\x47\x11\x28\x8D\x6B\x0C"
			  "\xE8\x77\x86\x1D\x9B\x10\xAC\xFC\xC1\x96\x43\x73\x82\x87\xDA\x00"
			  "\x52\x82\xF3\xFC\x82\xF9\xF5\x0A\xA6\x81\xF2\xF5\x5F\xE1\x80\x9C"
			  "\x9E\x23\xA3\xA5\x9E\x51\xC2\xE8\x94\xF7\x18\x37\x2F\x9F\xA1\x56"
			  "\x4B\x47\xAB\x3F\x43\xF0\x74\x7A\x17\x83\x9E\x93\x33\x69\xB6\x77"
			  "\x80\x53\xE1\x76\x4F\x52\xC5\xF3\x19\xE3\x3C\x8B\x25\x67\x8F\x72"
			  "\x33\x2E\x33\xCC\xA9\x7C\x68\xF1\x9E\x05\x8E\x70\xC3\x14\x10\xDF"
			  "\x4D\xE7\xE0\x81\x69\xD6\x09\x6B\x7B\x4E\xA4\x82\x71\xEB\x68\x4F"
			  "\xEE\x9F\xC8\xB5\x61\xC3\xFE\xE2\xDC\xE8\x3D\x09\x2B\x14\x2B\xEC"
			  "\x14\x78\xD2\x6B\x48\xC3\xC6\xE5\x97\xA7\xB2\xE4\x40\x27\xE1\xEC"
			  "\xA2\x31\x78\xD3\xAF\xCC\x67\xBB\x53\x0A\x52\x9C\x7E\x13\x36\xE1"
			  "\xAD\xAE\x74\xEF\x0B\xE9\xCD\x61\xE9\x1C\x6A\xEA\x57\xF7\xCC\xB2"
			  "\x3B\x64\xB2\xF8\x48\x61\xCE\x15\x92\x09\xFE\xF7\xA8\x97\xA1\x6A"
			  "\x87\x1A\xA9\x9E\x63\xA5\x12\x6D\xF2\xB0\x33\x87\xE4\x2C\x3D\x18"
};

xof_test_vector_t shake_128_255 = {
	.alg = XOF_SHAKE_128, .len = 255,
	.seed	= "\x3A\x3A\x81\x9C\x48\xEF\xDE\x2A\xD9\x14\xFB\xF0\x0E\x18\xAB\x6B"
			  "\xC4\xF1\x45\x13\xAB\x27\xD0\xC1\x78\xA1\x88\xB6\x14\x31\xE7\xF5"
			  "\x62\x3C\xB6\x6B\x23\x34\x67\x75\xD3\x86\xB5\x0E\x98\x2C\x49\x3A"
			  "\xDB\xBF\xC5\x4B\x9A\x3C\xD3\x83\x38\x23\x36\xA1\xA0\xB2\x15\x0A"
			  "\x15\x35\x8F\x33\x6D\x03\xAE\x18\xF6\x66\xC7\x57\x3D\x55\xC4\xFD"
			  "\x18\x1C\x29\xE6\xCC\xFD\xE6\x3E\xA3\x5F\x0A\xDF\x58\x85\xCF\xC0"
			  "\xA3\xD8\x4A\x2B\x2E\x4D\xD2\x44\x96\xDB\x78\x9E\x66\x31\x70\xCE"
			  "\xF7\x47\x98\xAA\x1B\xBC\xD4\x57\x4E\xA0\xBB\xA4\x04\x89\xD7\x64"
			  "\xB2\xF8\x3A\xAD\xC6\x6B\x14\x8B\x4A\x0C\xD9\x52\x46\xC1\x27\xD5"
			  "\x87\x1C\x4F\x11\x41\x86\x90\xA5\xDD\xF0\x12\x46\xA0\xC8\x0A\x43"
			  "\xC7\x00\x88\xB6\x18\x36\x39\xDC\xFD\xA4\x12\x5B\xD1\x13\xA8\xF4"
			  "\x9E\xE2\x3E\xD3\x06\xFA\xAC\x57\x6C\x3F\xB0\xC1\xE2\x56\x67\x1D"
			  "\x81\x7F\xC2\x53\x4A\x52\xF5\xB4\x39\xF7\x2E\x42\x4D\xE3\x76\xF4"
			  "\xC5\x65\xCC\xA8\x23\x07\xDD\x9E\xF7\x6D\xA5\xB7\xC4\xEB\x7E\x08"
			  "\x51\x72\xE3\x28\x80\x7C\x02\xD0\x11\xFF\xBF\x33\x78\x53\x78\xD7"
			  "\x9D\xC2\x66\xF6\xA5\xBE\x6B\xB0\xE4\xA9\x2E\xCE\xEB\xAE\xB1",
	.out_len = 512,
	.out	= "\x14\x23\x6E\x75\xB9\x78\x4D\xF4\xF5\x79\x35\xF9\x45\x35\x6C\xBE"
			  "\x38\x3F\xE5\x13\xED\x30\x28\x6F\x91\x06\x07\x59\xBC\xB0\xEF\x4B"
			  "\xAA\xC8\x58\xEC\xAE\x7C\x6E\x7E\xDD\x49\x8F\x01\xA0\x82\xB6\x3F"
			  "\xA5\x7D\x22\x54\x02\x31\xE2\xE2\x5C\x83\xEF\xB3\xB3\xF2\x95\x3A"
			  "\x5F\x67\x45\x02\xAB\x63\x52\x26\x44\x6B\x84\x93\x76\x43\xDC\xD5"
			  "\x78\x9E\xE7\x3F\x1D\x73\x4B\xC8\xFE\x5F\x7F\x08\x83\xAB\x10\x96"
			  "\x1B\x9A\x31\xFF\x60\xDE\xE1\x61\x59\xBC\x69\x82\xEF\xB0\x85\x45"
			  "\x98\x4B\xF7\x1F\xED\x1C\x4C\xD8\x1C\x09\x14\xB4\xC1\x9F\xCF\xEE"
			  "\xF5\x4A\xF4\xBB\xE3\x72\xF1\x8C\xFC\xD3\xA1\x86\x57\xF5\xB9\x45"
			  "\x0F\x99\xA7\x8F\x0F\xA2\xC3\xCD\xCA\x74\x61\xC4\xED\x75\x69\x53"
			  "\x68\x83\xB6\x6C\xD8\x7E\x9C\x20\x09\x62\x90\x2E\xAA\x16\xA5\x4D"
			  "\xB6\xA0\xA5\xCC\x26\xD8\x89\x03\x8C\x07\x60\x81\x0B\x5B\xB4\xF3"
			  "\x3F\x1E\x5D\x63\x9B\x6F\x9B\xC7\xCA\x62\xBA\x6F\x8C\x9F\x8D\xE7"
			  "\x70\x26\x0A\xFE\x47\xF4\xE0\xF8\x2F\x10\x21\x98\xEB\xA2\x7F\x54"
			  "\x32\x52\xAC\x8D\xDD\x83\xE1\xB8\xDB\x0A\x91\xAC\x65\x63\x3F\xD1"
			  "\x2A\x55\x0E\xBE\x96\xF9\x3A\xA6\x70\x4E\xD5\x90\x5C\x23\x4F\xA6"
			  "\xD9\x20\x39\x10\xCB\xD0\x2D\xE1\x66\xC4\xC3\x34\x8F\xB8\x1E\xF7"
			  "\xB8\x4A\xE1\x45\x5F\xE3\x18\xB5\xFD\x17\x08\x83\xF4\x9B\xA2\xF2"
			  "\x42\x89\xC4\x79\xA2\xC7\x53\x14\x06\xBA\x98\x9B\xEA\xEF\x3A\x79"
			  "\xF6\x59\x02\x86\x42\xE9\xB0\x33\xF7\xDE\xB9\xEC\xEC\x3A\x7A\x9F"
			  "\x1D\xBD\x24\x51\xFC\xB4\x7C\x81\xE2\x1E\x91\xD2\x0B\x92\x4C\x6B"
			  "\xD0\x4C\x1F\x0B\x27\x10\xD2\xE5\x70\xCD\x24\xBA\xD5\xB5\xDE\x4E"
			  "\x49\xAA\x80\xB6\xAD\xD5\x50\x7B\x4D\x2E\x51\x03\x70\xC7\xAF\xA8"
			  "\x14\xD7\xE1\xA7\xE2\x78\xE5\x3D\x7C\xCF\x49\xA0\xA8\x66\xCA\x3A"
			  "\x7B\x5B\xB7\x1E\xF3\x42\x5E\x46\x0F\xEE\xB2\x91\x49\xF2\x17\x06"
			  "\x66\x13\x69\x5F\x85\x50\x6A\x09\x46\xCF\x68\x97\x9F\x04\xAE\x07"
			  "\x3A\xF8\x02\x89\x76\xBF\x0C\x5B\xDC\x22\x12\xE8\xC3\x64\x58\x3D"
			  "\xE9\xFB\xD0\x3B\x34\xDD\xEE\x5E\xC4\xCF\xA8\xED\x8C\xE5\x92\x97"
			  "\x1D\x01\x08\xFA\xF7\x6C\x89\x40\xE2\x5E\x6C\x5F\x86\x55\x84\xC3"
			  "\x4A\x23\x3C\x14\xF0\x05\x32\x67\x3F\xDB\xE3\x88\xCC\x7E\x98\xA5"
			  "\xB8\x67\xB1\xC5\x91\x30\x7A\x90\x15\x11\x2B\x56\x7F\xF6\xB4\xF3"
			  "\x18\x11\x41\x11\xFC\x95\xE5\xBD\x7C\x9C\x60\xB7\x4C\x1F\x87\x25"
};

/**
 * SHAKE-256 vectors from "https://github.com/gvanas/KeccakCodePackage/"
 */
xof_test_vector_t shake_256_0 = {
	.alg = XOF_SHAKE_256, .len = 0,
	.seed	= "",
	.out_len = 512,
	.out	= "\x46\xB9\xDD\x2B\x0B\xA8\x8D\x13\x23\x3B\x3F\xEB\x74\x3E\xEB\x24"
			  "\x3F\xCD\x52\xEA\x62\xB8\x1B\x82\xB5\x0C\x27\x64\x6E\xD5\x76\x2F"
			  "\xD7\x5D\xC4\xDD\xD8\xC0\xF2\x00\xCB\x05\x01\x9D\x67\xB5\x92\xF6"
			  "\xFC\x82\x1C\x49\x47\x9A\xB4\x86\x40\x29\x2E\xAC\xB3\xB7\xC4\xBE"
			  "\x14\x1E\x96\x61\x6F\xB1\x39\x57\x69\x2C\xC7\xED\xD0\xB4\x5A\xE3"
			  "\xDC\x07\x22\x3C\x8E\x92\x93\x7B\xEF\x84\xBC\x0E\xAB\x86\x28\x53"
			  "\x34\x9E\xC7\x55\x46\xF5\x8F\xB7\xC2\x77\x5C\x38\x46\x2C\x50\x10"
			  "\xD8\x46\xC1\x85\xC1\x51\x11\xE5\x95\x52\x2A\x6B\xCD\x16\xCF\x86"
			  "\xF3\xD1\x22\x10\x9E\x3B\x1F\xDD\x94\x3B\x6A\xEC\x46\x8A\x2D\x62"
			  "\x1A\x7C\x06\xC6\xA9\x57\xC6\x2B\x54\xDA\xFC\x3B\xE8\x75\x67\xD6"
			  "\x77\x23\x13\x95\xF6\x14\x72\x93\xB6\x8C\xEA\xB7\xA9\xE0\xC5\x8D"
			  "\x86\x4E\x8E\xFD\xE4\xE1\xB9\xA4\x6C\xBE\x85\x47\x13\x67\x2F\x5C"
			  "\xAA\xAE\x31\x4E\xD9\x08\x3D\xAB\x4B\x09\x9F\x8E\x30\x0F\x01\xB8"
			  "\x65\x0F\x1F\x4B\x1D\x8F\xCF\x3F\x3C\xB5\x3F\xB8\xE9\xEB\x2E\xA2"
			  "\x03\xBD\xC9\x70\xF5\x0A\xE5\x54\x28\xA9\x1F\x7F\x53\xAC\x26\x6B"
			  "\x28\x41\x9C\x37\x78\xA1\x5F\xD2\x48\xD3\x39\xED\xE7\x85\xFB\x7F"
			  "\x5A\x1A\xAA\x96\xD3\x13\xEA\xCC\x89\x09\x36\xC1\x73\xCD\xCD\x0F"
			  "\xAB\x88\x2C\x45\x75\x5F\xEB\x3A\xED\x96\xD4\x77\xFF\x96\x39\x0B"
			  "\xF9\xA6\x6D\x13\x68\xB2\x08\xE2\x1F\x7C\x10\xD0\x4A\x3D\xBD\x4E"
			  "\x36\x06\x33\xE5\xDB\x4B\x60\x26\x01\xC1\x4C\xEA\x73\x7D\xB3\xDC"
			  "\xF7\x22\x63\x2C\xC7\x78\x51\xCB\xDD\xE2\xAA\xF0\xA3\x3A\x07\xB3"
			  "\x73\x44\x5D\xF4\x90\xCC\x8F\xC1\xE4\x16\x0F\xF1\x18\x37\x8F\x11"
			  "\xF0\x47\x7D\xE0\x55\xA8\x1A\x9E\xDA\x57\xA4\xA2\xCF\xB0\xC8\x39"
			  "\x29\xD3\x10\x91\x2F\x72\x9E\xC6\xCF\xA3\x6C\x6A\xC6\xA7\x58\x37"
			  "\x14\x30\x45\xD7\x91\xCC\x85\xEF\xF5\xB2\x19\x32\xF2\x38\x61\xBC"
			  "\xF2\x3A\x52\xB5\xDA\x67\xEA\xF7\xBA\xAE\x0F\x5F\xB1\x36\x9D\xB7"
			  "\x8F\x3A\xC4\x5F\x8C\x4A\xC5\x67\x1D\x85\x73\x5C\xDD\xDB\x09\xD2"
			  "\xB1\xE3\x4A\x1F\xC0\x66\xFF\x4A\x16\x2C\xB2\x63\xD6\x54\x12\x74"
			  "\xAE\x2F\xCC\x86\x5F\x61\x8A\xBE\x27\xC1\x24\xCD\x8B\x07\x4C\xCD"
			  "\x51\x63\x01\xB9\x18\x75\x82\x4D\x09\x95\x8F\x34\x1E\xF2\x74\xBD"
			  "\xAB\x0B\xAE\x31\x63\x39\x89\x43\x04\xE3\x58\x77\xB0\xC2\x8A\x9B"
			  "\x1F\xD1\x66\xC7\x96\xB9\xCC\x25\x8A\x06\x4A\x8F\x57\xE2\x7F\x2A"
};

xof_test_vector_t shake_256_64 = {
	.alg = XOF_SHAKE_256, .len = 64,
	.seed	= "\xE9\x26\xAE\x8B\x0A\xF6\xE5\x31\x76\xDB\xFF\xCC\x2A\x6B\x88\xC6"
			  "\xBD\x76\x5F\x93\x9D\x3D\x17\x8A\x9B\xDE\x9E\xF3\xAA\x13\x1C\x61"
			  "\xE3\x1C\x1E\x42\xCD\xFA\xF4\xB4\xDC\xDE\x57\x9A\x37\xE1\x50\xEF"
			  "\xBE\xF5\x55\x5B\x4C\x1C\xB4\x04\x39\xD8\x35\xA7\x24\xE2\xFA\xE7",
	.out_len = 512,
	.out	= "\x77\xB7\x49\x6E\xD0\x8C\x39\x33\xBD\x75\x98\x3C\x0C\x04\x94\xBD"
			  "\xD8\x26\x24\x93\xA4\xB5\x5D\xDC\xCC\x64\x16\x7E\x67\xEA\xC0\xF6"
			  "\xE6\x30\x7A\xCC\x15\xC3\x3F\x39\x63\x74\x4E\x26\xCA\x6C\x50\x4D"
			  "\x39\x3B\x3E\xE8\x16\x5E\x4D\x49\xEB\x3B\x6E\x64\x92\x07\x65\x30"
			  "\x48\xF8\xB8\x22\xFF\x88\x4D\xC7\x49\x37\x44\x3B\x1C\x4A\x88\x8C"
              "\x7A\x76\x8C\x63\xD5\xB5\xD2\x9E\x74\x46\x87\x39\x23\xB9\xD7\xA5"
			  "\x6F\xA5\xD9\xE9\x07\x60\xAB\x86\xD5\x71\x8E\x34\x64\x82\x1B\x79"
			  "\xEB\x46\xD1\x69\x14\x1F\xF1\x61\x20\xBF\xB6\x50\xC7\x6D\x4B\x3E"
			  "\x5B\x3F\x6C\xE6\x1F\xEB\xDB\xE0\x9A\xED\x7F\x4C\x91\x06\x6D\x90"
			  "\x3A\xF6\xE5\x65\x31\xE8\xFF\x71\x54\x95\x08\xB6\xE4\x20\xCA\xC6"
			  "\xBE\xDF\xE0\xCB\xEA\xE6\xBC\x22\x84\x76\xBC\x8C\x00\xEA\xE4\x3D"
			  "\x40\xC8\x2C\xBD\xF6\xB4\x60\xC3\x76\xD7\xC1\x16\x48\xEB\x28\x15"
			  "\xB6\x50\x6A\xBD\x43\x39\xB2\x5D\x58\xD4\x5C\xDD\x0A\x0B\x9E\x35"
			  "\xA8\x8E\x25\x1F\xDC\x34\xD4\x81\x0D\x65\x9D\x17\x9F\x59\xEB\xD0"
			  "\x37\x17\xFD\x31\xA6\x39\x4C\xE1\x2C\xD5\x56\x90\x66\xE1\x38\x88"
			  "\x5C\xB2\xBD\xEB\xBA\x06\x36\x75\x57\xCE\x84\x9E\xB8\x69\xF3\xCA"
			  "\xC3\x88\x00\xD5\x1C\x22\xB6\x66\xAE\x27\x01\xE5\x80\x79\x63\x94"
			  "\xDF\xA0\x2F\x49\x10\xBF\x5F\x86\xAA\xB5\x39\x51\x23\x33\x64\xEA"
			  "\x20\xCD\xA3\x5A\xFB\xAB\x44\x5B\xE7\xF6\x86\x64\x38\x56\xF8\x25"
			  "\x39\x4B\xE7\xB4\xB6\xD2\xC9\x18\xD0\x15\x1F\x46\xFB\x9A\xEE\x8A"
			  "\x7B\xA2\xD7\x06\xE4\x8C\xB0\xBC\x42\x9B\x06\x42\x62\xC1\xA0\xEB"
			  "\x35\x24\xFF\x14\x63\x2F\x51\x84\x57\x5C\x15\xF6\xF4\xA3\x44\x6E"
			  "\x93\xCB\x4E\x86\xB6\xA9\x31\xBA\x26\x84\x09\xCE\x30\xB4\x59\x5F"
			  "\xD2\x05\x9A\x27\x18\x3B\x3B\xA8\xD0\xAC\xE8\xE4\x82\x86\x6D\x5C"
			  "\x7D\x5B\x03\xDB\x8D\xBD\x24\xB9\x9D\x59\xEB\x6E\xEF\xFD\x20\x9E"
			  "\x12\x45\x35\xD1\x54\xB9\x8F\x99\x91\xD8\x4F\xE1\xAA\x76\x3C\x51"
			  "\x33\xD4\x1E\xCC\x23\x39\x30\x95\x7D\xCE\xB7\x89\x6A\xF7\x0F\x73"
			  "\x5A\x2F\x5C\x1E\x79\x48\x0A\xFD\x50\x94\x3B\xC5\x01\x4B\xCF\x0A"
			  "\x73\x54\xAA\x7F\x71\x31\x63\xB5\x5A\x1E\x41\xBD\xD0\x5F\xBB\xA9"
			  "\xC1\xDB\x2C\x69\x04\x3E\xD9\xEE\xA4\xFA\x45\xC9\x90\xCC\xB4\xA8"
			  "\xDC\x41\xAF\xAB\x18\x16\x40\x18\xE5\x4C\x47\xAC\x5B\xD6\x98\x0F"
			  "\xD7\x96\xAC\xF0\xDD\xB4\x2C\x70\x42\xA4\x87\x7E\x8B\xE3\xDE\x29"
};

xof_test_vector_t shake_256_135 = {
	.alg = XOF_SHAKE_256, .len = 135,
	.seed	= "\xB7\x71\xD5\xCE\xF5\xD1\xA4\x1A\x93\xD1\x56\x43\xD7\x18\x1D\x2A"
			  "\x2E\xF0\xA8\xE8\x4D\x91\x81\x2F\x20\xED\x21\xF1\x47\xBE\xF7\x32"
			  "\xBF\x3A\x60\xEF\x40\x67\xC3\x73\x4B\x85\xBC\x8C\xD4\x71\x78\x0F"
			  "\x10\xDC\x9E\x82\x91\xB5\x83\x39\xA6\x77\xB9\x60\x21\x8F\x71\xE7"
			  "\x93\xF2\x79\x7A\xEA\x34\x94\x06\x51\x28\x29\x06\x5D\x37\xBB\x55"
			  "\xEA\x79\x6F\xA4\xF5\x6F\xD8\x89\x6B\x49\xB2\xCD\x19\xB4\x32\x15"
			  "\xAD\x96\x7C\x71\x2B\x24\xE5\x03\x2D\x06\x52\x32\xE0\x2C\x12\x74"
			  "\x09\xD2\xED\x41\x46\xB9\xD7\x5D\x76\x3D\x52\xDB\x98\xD9\x49\xD3"
			  "\xB0\xFE\xD6\xA8\x05\x2F\xBB",
	.out_len = 512,
	.out	= "\x6C\x60\x95\x5D\xCB\x8A\x66\x3B\x6D\xC7\xF5\xEF\x7E\x06\x9C\xA8"
			  "\xFE\x3D\xA9\x9A\x66\xDF\x65\x96\x92\x5D\x55\x7F\xED\x91\xF4\x70"
			  "\x91\x40\x7D\x6F\xDE\x32\x02\x3B\x57\xE2\xEE\x4C\x6A\xC9\x7B\x07"
			  "\x76\x24\xFA\xC2\x5F\x6E\x13\xF4\x19\x16\x96\xB4\x0A\x4D\xF7\x5F"
			  "\x61\xCD\x55\x21\xD9\x82\xC6\xD0\x9D\x83\x42\xC1\x7A\x36\x6E\xC6"
			  "\x34\x6E\x35\x28\xB2\x6C\xFF\x91\x5B\xE9\x44\x2B\x9E\xBC\xC3\x0F"
			  "\xF2\xF6\xAD\xD0\xE8\x2B\xA9\x04\xC7\x37\x00\xCC\x99\xAC\xFF\x48"
			  "\x0C\xAF\x04\x87\xCE\xE5\x4C\xBA\x37\x53\xB6\xA5\xDD\x6F\x0D\xFE"
			  "\x65\x71\xF0\x11\x5E\x87\x37\xB0\x71\x03\x10\x23\xB6\xBB\x0D\x79"
			  "\x86\x4C\x3F\x33\x16\x2E\x78\x26\x9C\xEE\x23\xFC\xE4\x7B\x91\xB4"
			  "\xFD\xF9\x1F\x98\x46\x4A\x1D\x21\xE7\x99\xD1\x7F\x76\xC1\xBB\x80"
			  "\x7D\xEE\x66\x7B\x0B\x27\x30\x54\xBE\x29\x82\x99\xBD\x12\xB7\xA8"
			  "\x0F\xB3\x54\xCE\x3E\x6D\x1A\xCF\x98\x44\x38\x79\xA5\x54\xEC\xA6"
			  "\xB9\x6D\xF0\x61\xD0\x4A\x11\x7C\x98\xAE\xEC\x1C\xDE\x1A\xFA\x9C"
			  "\xEF\x62\xDD\x68\x6D\xA9\x1B\xB2\xB1\xF1\x23\x79\xBB\xDC\x9F\xA3"
			  "\x2A\x6B\x69\x98\xB7\x7E\x8E\xB0\xB5\x05\x07\x86\x2A\xFA\x77\x99"
			  "\xD0\x18\xE2\x72\x09\x1F\x51\xCA\xDD\x81\xAD\xB5\x87\xEF\x67\xBA"
			  "\x67\x61\x8C\x45\xD1\xF3\xD5\x59\xDB\xD2\x99\xAB\xC2\x6E\xC7\x12"
			  "\xDA\x8F\xA3\x4B\xA3\x3B\xFF\x40\x0D\x1F\x0F\x8B\x63\x45\xCF\x57"
			  "\x26\x9B\x85\x85\x78\xC0\x07\x2A\x91\xA6\x3E\xF8\x5F\x9D\x37\x89"
			  "\x00\xCD\x1A\x55\xD2\xBD\x46\x30\xDB\x82\x9E\xB4\x84\xD8\x9C\xE7"
			  "\xA4\x14\xAC\xA1\x73\xC5\x25\x34\xAD\x5F\x93\x55\xE8\x0E\x39\x5E"
			  "\x79\x15\x6D\x75\x1A\x93\x0F\x7F\x8B\x5D\x9F\x4D\x5A\x2C\x9A\x75"
			  "\x37\x23\x08\x3C\x5E\x8E\xC6\xCB\x24\xD8\xEF\x93\xC8\xFE\xF2\xD1"
			  "\xBE\x4E\xCA\x22\x2C\x6E\x6C\x2A\xCF\xD6\x84\x89\x3C\xEA\x65\xCB"
			  "\xF5\xB0\x96\xB3\xD8\x66\x00\x71\x36\x12\x6A\x33\xEF\x49\x6B\xF2"
			  "\x31\x0F\x29\x3B\xFA\x4C\x93\xAB\x82\x68\x21\xE2\xB9\x32\x59\xC4"
			  "\x64\xE0\xAE\xB0\x6D\x6D\xF8\xFF\xA3\x0B\x1C\x1E\x7E\x38\x4C\x7E"
			  "\x42\x7A\x2B\xA3\xD9\x9F\xF8\xA6\x66\x38\x0C\x5C\x1B\x67\x8F\x74"
			  "\x2C\x57\xB0\xC3\xB0\x88\x49\xFD\x65\x30\x0D\xF1\x34\x99\xDD\x89"
			  "\x4E\xFC\x33\x11\x6E\x7D\x07\x74\x06\x43\x31\xFD\xD4\x07\x48\x74"
			  "\x17\xD1\x3B\xBA\x42\x85\x29\x9A\xF6\x50\xD3\x06\x5D\x95\x11\x31"
};

xof_test_vector_t shake_256_136 = {
	.alg = XOF_SHAKE_256, .len = 136,
	.seed	= "\xB3\x2D\x95\xB0\xB9\xAA\xD2\xA8\x81\x6D\xE6\xD0\x6D\x1F\x86\x00"
			  "\x85\x05\xBD\x8C\x14\x12\x4F\x6E\x9A\x16\x3B\x5A\x2A\xDE\x55\xF8"
			  "\x35\xD0\xEC\x38\x80\xEF\x50\x70\x0D\x3B\x25\xE4\x2C\xC0\xAF\x05"
			  "\x0C\xCD\x1B\xE5\xE5\x55\xB2\x30\x87\xE0\x4D\x7B\xF9\x81\x36\x22"
			  "\x78\x0C\x73\x13\xA1\x95\x4F\x87\x40\xB6\xEE\x2D\x3F\x71\xF7\x68"
			  "\xDD\x41\x7F\x52\x04\x82\xBD\x3A\x08\xD4\xF2\x22\xB4\xEE\x9D\xBD"
			  "\x01\x54\x47\xB3\x35\x07\xDD\x50\xF3\xAB\x42\x47\xC5\xDE\x9A\x8A"
			  "\xBD\x62\xA8\xDE\xCE\xA0\x1E\x3B\x87\xC8\xB9\x27\xF5\xB0\x8B\xEB"
			  "\x37\x67\x4C\x6F\x8E\x38\x0C\x04",
	.out_len = 512,
	.out	= "\xCC\x2E\xAA\x04\xEE\xF8\x47\x9C\xDA\xE8\x56\x6E\xB8\xFF\xA1\x10"
			  "\x0A\x40\x79\x95\xBF\x99\x9A\xE9\x7E\xDE\x52\x66\x81\xDC\x34\x90"
			  "\x61\x6F\x28\x44\x2D\x20\xDA\x92\x12\x4C\xE0\x81\x58\x8B\x81\x49"
			  "\x1A\xED\xF6\x5C\xAA\xF0\xD2\x7E\x82\xA4\xB0\xE1\xD1\xCA\xB2\x38"
			  "\x33\x32\x8F\x1B\x8D\xA4\x30\xC8\xA0\x87\x66\xA8\x63\x70\xFA\x84"
			  "\x8A\x79\xB5\x99\x8D\xB3\xCF\xFD\x05\x7B\x96\xE1\xE2\xEE\x0E\xF2"
			  "\x29\xEC\xA1\x33\xC1\x55\x48\xF9\x83\x99\x02\x04\x37\x30\xE4\x4B"
			  "\xC5\x2C\x39\xFA\xDC\x1D\xDE\xEA\xD9\x5F\x99\x39\xF2\x20\xCA\x30"
			  "\x06\x61\x54\x0D\xF7\xED\xD9\xAF\x37\x8A\x5D\x4A\x19\xB2\xB9\x3E"
			  "\x6C\x78\xF4\x9C\x35\x33\x43\xA0\xB5\xF1\x19\x13\x2B\x53\x12\xD0"
			  "\x04\x83\x1D\x01\x76\x9A\x31\x6D\x2F\x51\xBF\x64\xCC\xB2\x0A\x21"
			  "\xC2\xCF\x7A\xC8\xFB\x6F\x6E\x90\x70\x61\x26\xBD\xAE\x06\x11\xDD"
			  "\x13\x96\x2E\x8B\x53\xD6\xEA\xE2\x6C\x7B\x0D\x25\x51\xDA\xF6\x24"
			  "\x8E\x9D\x65\x81\x73\x82\xB0\x4D\x23\x39\x2D\x10\x8E\x4D\x34\x43"
			  "\xDE\x5A\xDC\x72\x73\xC7\x21\xA8\xF8\x32\x0E\xCF\xE8\x17\x7A\xC0"
			  "\x67\xCA\x8A\x50\x16\x9A\x6E\x73\x00\x0E\xBC\xDC\x1E\x4E\xE6\x33"
			  "\x9F\xC8\x67\xC3\xD7\xAE\xAB\x84\x14\x63\x98\xD7\xBA\xDE\x12\x1D"
			  "\x19\x89\xFA\x45\x73\x35\x56\x4E\x97\x57\x70\xA3\xA0\x02\x59\xCA"
			  "\x08\x70\x61\x08\x26\x1A\xA2\xD3\x4D\xE0\x0F\x8C\xAC\x7D\x45\xD3"
			  "\x5E\x5A\xA6\x3E\xA6\x9E\x1D\x1A\x2F\x7D\xAB\x39\x00\xD5\x1E\x0B"
			  "\xC6\x53\x48\xA2\x55\x54\x00\x70\x39\xA5\x2C\x3C\x30\x99\x80\xD1"
			  "\x7C\xAD\x20\xF1\x15\x63\x10\xA3\x9C\xD3\x93\x76\x0C\xFE\x58\xF6"
			  "\xF8\xAD\xE4\x21\x31\x28\x82\x80\xA3\x5E\x1D\xB8\x70\x81\x83\xB9"
			  "\x1C\xFA\xF5\x82\x7E\x96\xB0\xF7\x74\xC4\x50\x93\xB4\x17\xAF\xF9"
			  "\xDD\x64\x17\xE5\x99\x64\xA0\x1B\xD2\xA6\x12\xFF\xCF\xBA\x18\xA0"
			  "\xF1\x93\xDB\x29\x7B\x9A\x6C\xC1\xD2\x70\xD9\x7A\xAE\x8F\x8A\x3A"
			  "\x6B\x26\x69\x5A\xB6\x64\x31\xC2\x02\xE1\x39\xD6\x3D\xD3\xA2\x47"
			  "\x78\x67\x6C\xEF\xE3\xE2\x1B\x02\xEC\x4E\x8F\x5C\xFD\x66\x58\x7A"
			  "\x12\xB4\x40\x78\xFC\xD3\x9E\xEE\x44\xBB\xEF\x4A\x94\x9A\x63\xC0"
			  "\xDF\xD5\x8C\xF2\xFB\x2C\xD5\xF0\x02\xE2\xB0\x21\x92\x66\xCF\xC0"
			  "\x31\x81\x74\x86\xDE\x70\xB4\x28\x5A\x8A\x70\xF3\xD3\x8A\x61\xD3"
			  "\x15\x5D\x99\xAA\xF4\xC2\x53\x90\xD7\x36\x45\xAB\x3E\x8D\x80\xF0"
};

xof_test_vector_t shake_256_255 = {
	.alg = XOF_SHAKE_256, .len = 255,
	.seed	= "\x3A\x3A\x81\x9C\x48\xEF\xDE\x2A\xD9\x14\xFB\xF0\x0E\x18\xAB\x6B"
			  "\xC4\xF1\x45\x13\xAB\x27\xD0\xC1\x78\xA1\x88\xB6\x14\x31\xE7\xF5"
			  "\x62\x3C\xB6\x6B\x23\x34\x67\x75\xD3\x86\xB5\x0E\x98\x2C\x49\x3A"
			  "\xDB\xBF\xC5\x4B\x9A\x3C\xD3\x83\x38\x23\x36\xA1\xA0\xB2\x15\x0A"
			  "\x15\x35\x8F\x33\x6D\x03\xAE\x18\xF6\x66\xC7\x57\x3D\x55\xC4\xFD"
			  "\x18\x1C\x29\xE6\xCC\xFD\xE6\x3E\xA3\x5F\x0A\xDF\x58\x85\xCF\xC0"
			  "\xA3\xD8\x4A\x2B\x2E\x4D\xD2\x44\x96\xDB\x78\x9E\x66\x31\x70\xCE"
			  "\xF7\x47\x98\xAA\x1B\xBC\xD4\x57\x4E\xA0\xBB\xA4\x04\x89\xD7\x64"
			  "\xB2\xF8\x3A\xAD\xC6\x6B\x14\x8B\x4A\x0C\xD9\x52\x46\xC1\x27\xD5"
			  "\x87\x1C\x4F\x11\x41\x86\x90\xA5\xDD\xF0\x12\x46\xA0\xC8\x0A\x43"
			  "\xC7\x00\x88\xB6\x18\x36\x39\xDC\xFD\xA4\x12\x5B\xD1\x13\xA8\xF4"
			  "\x9E\xE2\x3E\xD3\x06\xFA\xAC\x57\x6C\x3F\xB0\xC1\xE2\x56\x67\x1D"
			  "\x81\x7F\xC2\x53\x4A\x52\xF5\xB4\x39\xF7\x2E\x42\x4D\xE3\x76\xF4"
			  "\xC5\x65\xCC\xA8\x23\x07\xDD\x9E\xF7\x6D\xA5\xB7\xC4\xEB\x7E\x08"
			  "\x51\x72\xE3\x28\x80\x7C\x02\xD0\x11\xFF\xBF\x33\x78\x53\x78\xD7"
			  "\x9D\xC2\x66\xF6\xA5\xBE\x6B\xB0\xE4\xA9\x2E\xCE\xEB\xAE\xB1",
	.out_len = 512,
	.out	= "\x8A\x51\x99\xB4\xA7\xE1\x33\xE2\x64\xA8\x62\x02\x72\x06\x55\x89"
			  "\x4D\x48\xCF\xF3\x44\xA9\x28\xCF\x83\x47\xF4\x83\x79\xCE\xF3\x47"
			  "\xDF\xC5\xBC\xFF\xAB\x99\xB2\x7B\x1F\x89\xAA\x27\x35\xE2\x3D\x30"
			  "\x08\x8F\xFA\x03\xB9\xED\xB0\x2B\x96\x35\x47\x0A\xB9\xF1\x03\x89"
			  "\x85\xD5\x5F\x9C\xA7\x74\x57\x2D\xD0\x06\x47\x0E\xA6\x51\x45\x46"
			  "\x96\x09\xF9\xFA\x08\x31\xBF\x1F\xFD\x84\x2D\xC2\x4A\xCA\xDE\x27"
			  "\xBD\x98\x16\xE3\xB5\xBF\x28\x76\xCB\x11\x22\x32\xA0\xEB\x44\x75"
			  "\xF1\xDF\xF9\xF5\xC7\x13\xD9\xFF\xD4\xCC\xB8\x9A\xE5\x60\x7F\xE3"
			  "\x57\x31\xDF\x06\x31\x79\x49\xEE\xF6\x46\xE9\x59\x1C\xF3\xBE\x53"
			  "\xAD\xD6\xB7\xDD\x2B\x60\x96\xE2\xB3\xFB\x06\xE6\x62\xEC\x8B\x2D"
			  "\x77\x42\x2D\xAA\xD9\x46\x3C\xD1\x55\x20\x4A\xCD\xBD\x38\xE3\x19"
			  "\x61\x3F\x39\xF9\x9B\x6D\xFB\x35\xCA\x93\x65\x16\x00\x66\xDB\x19"
			  "\x83\x58\x88\xC2\x24\x1F\xF9\xA7\x31\xA4\xAC\xBB\x56\x63\x72\x7A"
			  "\xAC\x34\xA4\x01\x24\x7F\xBA\xA7\x49\x9E\x7D\x5E\xE5\xB6\x9D\x31"
			  "\x02\x5E\x63\xD0\x4C\x35\xC7\x98\xBC\xA1\x26\x2D\x56\x73\xA9\xCF"
			  "\x09\x30\xB5\xAD\x89\xBD\x48\x55\x99\xDC\x18\x45\x28\xDA\x47\x90"
			  "\xF0\x88\xEB\xD1\x70\xB6\x35\xD9\x58\x16\x32\xD2\xFF\x90\xDB\x79"
			  "\x66\x5C\xED\x43\x00\x89\xAF\x13\xC9\xF2\x1F\x6D\x44\x3A\x81\x80"
			  "\x64\xF1\x7A\xEC\x9E\x9C\x54\x57\x00\x1F\xA8\xDC\x6A\xFB\xAD\xBE"
			  "\x31\x38\xF3\x88\xD8\x9D\x0E\x6F\x22\xF6\x66\x71\x25\x5B\x21\x07"
			  "\x54\xED\x63\xD8\x1D\xCE\x75\xCE\x8F\x18\x9B\x53\x4E\x6D\x6B\x35"
			  "\x39\xAA\x51\xE8\x37\xC4\x2D\xF9\xDF\x59\xC7\x1E\x61\x71\xCD\x49"
			  "\x02\xFE\x1B\xDC\x73\xFB\x17\x75\xB5\xC7\x54\xA1\xED\x4E\xA7\xF3"
			  "\x10\x5F\xC5\x43\xEE\x04\x18\xDA\xD2\x56\xF3\xF6\x11\x8E\xA7\x71"
			  "\x14\xA1\x6C\x15\x35\x5B\x42\x87\x7A\x1D\xB2\xA7\xDF\x0E\x15\x5A"
			  "\xE1\xD8\x67\x0A\xBC\xEC\x34\x50\xF4\xE2\xEE\xC9\x83\x8F\x89\x54"
			  "\x23\xEF\x63\xD2\x61\x13\x8B\xAA\xF5\xD9\xF1\x04\xCB\x5A\x95\x7A"
			  "\xEA\x06\xC0\xB9\xB8\xC7\x8B\x0D\x44\x17\x96\xDC\x03\x50\xDD\xEA"
			  "\xBB\x78\xA3\x3B\x6F\x1F\x9E\x68\xED\xE3\xD1\x80\x5C\x7B\x7E\x2C"
			  "\xFD\x54\xE0\xFA\xD6\x2F\x0D\x8C\xA6\x7A\x77\x5D\xC4\x54\x6A\xF9"
			  "\x09\x6F\x2E\xDB\x22\x1D\xB4\x28\x43\xD6\x53\x27\x86\x12\x82\xDC"
			  "\x94\x6A\x0B\xA0\x1A\x11\x86\x3A\xB2\xD1\xDF\xD1\x6E\x39\x73\xD4"
};
