%% Copyright (C) 2024 by sysmocom - s.f.m.c. GmbH <info@sysmocom.de>
%% Author: Vadim Yanitskiy <vyanitskiy@sysmocom.de>
%%
%% All Rights Reserved
%%
%% SPDX-License-Identifier: AGPL-3.0-or-later
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU Affero General Public License as
%% published by the Free Software Foundation; either version 3 of the
%% License, or (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU Affero General Public License
%% along with this program.  If not, see <https://www.gnu.org/licenses/>.
%%
%% Additional Permission under GNU AGPL version 3 section 7:
%%
%% If you modify this Program, or any covered work, by linking or
%% combining it with runtime libraries of Erlang/OTP as released by
%% Ericsson on https://www.erlang.org (or a modified version of these
%% libraries), containing parts covered by the terms of the Erlang Public
%% License (https://www.erlang.org/EPLICENSE), the licensors of this
%% Program grant you additional permission to convey the resulting work
%% without the need to license the runtime libraries of Erlang/OTP under
%% the GNU Affero General Public License. Corresponding Source for a
%% non-source form of such a combination shall include the source code
%% for the parts of the runtime libraries of Erlang/OTP used as well as
%% that of the covered work.

-module(sctp_client).

-export([connect/1]).

-include_lib("kernel/include/logger.hrl").
-include_lib("kernel/include/inet.hrl").
-include_lib("kernel/include/inet_sctp.hrl").


-type cfg() :: #{laddr => any | string() | inet:ip_address(),
                 raddr => loopback | string() | inet:ip_address(),
                 rport => inet:port_number(),
                 sockopts => [gen_sctp:option()]
                }.

-type connect_result() :: {ok, gen_sctp:sctp_socket()}.

-export_type([cfg/0,
              connect_result/0]).


%% ------------------------------------------------------------------
%% behavior callbacks
%% ------------------------------------------------------------------

-callback start_link(Aid, Priv) -> Result
    when Aid :: gen_sctp:assoc_id(),
         Priv :: term(),
         Result :: {ok, pid()} | term().


-callback send_data(Pid, Data) -> Result
    when Pid :: pid(),
         Data :: binary(),
         Result :: term().


-callback shutdown(Pid) -> Result
    when Pid :: pid(),
         Result :: term().


%% ------------------------------------------------------------------
%% public API
%% ------------------------------------------------------------------

-spec connect(cfg()) -> connect_result().
connect(Cfg) ->
    SockOpts = maps:get(sockopts, Cfg, []), %% additional sockopts
    LAddr = sctp_common:parse_addr(maps:get(laddr, Cfg, any)),
    RAddr = sctp_common:parse_addr(maps:get(raddr, Cfg, localhost)),
    RPort = maps:get(rport, Cfg),
    {ok, Sock} = gen_sctp:open([{ip, LAddr},
                                {type, seqpacket},
                                {active, true}] ++ SockOpts),
    gen_sctp:connect_init(Sock, RAddr, RPort, []),
    {ok, Sock}.

%% vim:set ts=4 sw=4 et:
