%% Copyright (C) 2025 by sysmocom - s.f.m.c. GmbH <info@sysmocom.de>
%% Author: Vadim Yanitskiy <vyanitskiy@sysmocom.de>
%%
%% All Rights Reserved
%%
%% SPDX-License-Identifier: AGPL-3.0-or-later
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU Affero General Public License as
%% published by the Free Software Foundation; either version 3 of the
%% License, or (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU Affero General Public License
%% along with this program.  If not, see <https://www.gnu.org/licenses/>.
%%
%% Additional Permission under GNU AGPL version 3 section 7:
%%
%% If you modify this Program, or any covered work, by linking or
%% combining it with runtime libraries of Erlang/OTP as released by
%% Ericsson on https://www.erlang.org (or a modified version of these
%% libraries), containing parts covered by the terms of the Erlang Public
%% License (https://www.erlang.org/EPLICENSE), the licensors of this
%% Program grant you additional permission to convey the resulting work
%% without the need to license the runtime libraries of Erlang/OTP under
%% the GNU Affero General Public License. Corresponding Source for a
%% non-source form of such a combination shall include the source code
%% for the parts of the runtime libraries of Erlang/OTP used as well as
%% that of the covered work.

-module(enb_uptime).

-export([start_link/0,
         genb_id_ind/2,
         shutdown/1]).

-include_lib("kernel/include/logger.hrl").

-include("s1gw_metrics.hrl").

-define(INTERVAL, 5_000).


%% ------------------------------------------------------------------
%% public API
%% ------------------------------------------------------------------

-spec start_link() -> pid().
start_link() ->
    T0 = erlang:monotonic_time(),
    spawn_link(fun() -> loop(#{time => T0}) end).


-spec genb_id_ind(pid(), string()) -> ok.
genb_id_ind(Pid, GlobalENBId) ->
    Pid ! {?FUNCTION_NAME, GlobalENBId},
    ok.


-spec shutdown(pid()) -> ok.
shutdown(Pid) ->
    Pid ! stop,
    ok.


%% ------------------------------------------------------------------
%% private API
%% ------------------------------------------------------------------

loop(#{genb_id := GlobalENBId, time := T0} = S) ->
    receive
        stop ->
            s1gw_metrics:ctr_reset(?S1GW_CTR_ENB_UPTIME(GlobalENBId)),
            ok;
        Event ->
            ?LOG_ERROR("Rx unexpected event: ~p", [Event])
    after ?INTERVAL ->
        T1 = erlang:monotonic_time(),
        Diff = erlang:convert_time_unit(T1 - T0, native, second),
        s1gw_metrics:ctr_inc(?S1GW_CTR_ENB_UPTIME(GlobalENBId), Diff),
        loop(S#{time => T1})
    end;

loop(#{} = S) ->
    receive
        {genb_id_ind, GlobalENBId} ->
            catch exometer:new(?S1GW_CTR_ENB_UPTIME(GlobalENBId), counter),
            loop(S#{genb_id => GlobalENBId});
        stop ->
            ok;
        Event ->
            ?LOG_ERROR("Rx unexpected event: ~p", [Event])
    end.


%% vim:set ts=4 sw=4 et:
