-module(s1ap_proxy_test).

-include_lib("eunit/include/eunit.hrl").
-include("s1gw_metrics.hrl").
-include("pfcp_mock.hrl").


-define(GlobalENBId, "001-01-0").

-define(_assertMetric(Name, Value),
        ?_assertEqual(Value, s1gw_metrics:get_current_value(Name))).

-define(_assertMetricENB(Name, Value),
        ?_assertMetric(s1gw_metrics:enb_metric(Name, ?GlobalENBId), Value)).


%% ------------------------------------------------------------------
%% setup functions
%% ------------------------------------------------------------------

-define(TC(Fun), {setup,
                  fun start/0,
                  fun stop/1,
                  Fun}).


start() ->
    pfcp_mock:mock_all(),
    exometer:start(),
    s1gw_metrics:init(),
    gtpu_kpi:start_link(#{enable => false}),
    {ok, Pid} = s1ap_proxy:start_link(#{addr => {127,0,0,0},
                                        port => 1337}),
    #{handler => Pid}.


stop(#{handler := Pid}) ->
    s1ap_proxy:shutdown(Pid),
    exometer:stop(),
    gtpu_kpi:shutdown(),
    pfcp_mock:unmock_all().


%% ------------------------------------------------------------------
%% testcase descriptions
%% ------------------------------------------------------------------

s1ap_proxy_test_() ->
    [{"S1 SETUP REQUEST/RESPONSE (unmodified)",
      ?TC(fun test_s1_setup/1)},
     {"E-RAB SETUP REQUEST/RESPONSE",
      ?TC(fun test_e_rab_setup/1)},
     {"E-RAB SETUP REQUEST (failure)",
      ?TC(fun test_e_rab_setup_req_fail/1)},
     {"E-RAB RELEASE COMMAND/RESPONSE",
      ?TC(fun test_e_rab_release_cmd/1)},
     {"E-RAB RELEASE INDICATION",
      ?TC(fun test_e_rab_release_ind/1)},
     {"E-RAB MODIFY REQUEST/RESPONSE (success)",
      ?TC(fun test_e_rab_modify_req_rsp/1)},
     {"E-RAB MODIFY REQUEST/RESPONSE (failure)",
      ?TC(fun test_e_rab_modify_req_rsp_fail/1)},
     {"E-RAB MODIFICATION INDICATION (modified)",
      ?TC(fun test_e_rab_modify_ind_cnf_modified/1)},
     {"E-RAB MODIFICATION INDICATION (not modified)",
      ?TC(fun test_e_rab_modify_ind_cnf_not_modified/1)},
     {"E-RAB MODIFICATION INDICATION (release)",
      ?TC(fun test_e_rab_modify_ind_cnf_release/1)},
     {"INITIAL CONTEXT SETUP REQUEST/RESPONSE",
      ?TC(fun test_initial_context_setup/1)},
     {"UE CONTEXT RELEASE REQUEST",
      ?TC(fun test_ue_ctx_release_req/1)},
     {"UE CONTEXT RELEASE COMMAND/COMPLETE",
      ?TC(fun test_ue_ctx_release_cmd/1)},
     {"ASN.1 parsing error (drop)",
      ?TC(fun test_drop_asn1_error/1)},
     {"PDU processing error (drop)",
      ?TC(fun test_drop_proc_error/1)}].


%% ------------------------------------------------------------------
%% actual testcases
%% ------------------------------------------------------------------

test_s1_setup(#{handler := Pid}) ->
    SetupReq = s1_setup_req_pdu(),
    SetupRsp = s1_setup_rsp_pdu(),
    %% Expect the PDUs to be proxied unmodified
    [?_assertEqual({forward, SetupReq}, s1ap_proxy:process_pdu(Pid, SetupReq)),
     ?_assertEqual({forward, SetupRsp}, s1ap_proxy:process_pdu(Pid, SetupRsp)),
     %% global counters
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 2),
     %% per-eNB counters
     ?_assertMetricENB(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 2),
     ?_assertMetricENB(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2),
     ?_assertMetricENB(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 2)].


test_e_rab_setup(#{handler := Pid}) ->
    %% [eNB <- MME] E-RAB SETUP REQUEST
    SetupReqIn = e_rab_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    SetupReqExp = e_rab_setup_req_pdu(?ADDR_A2U, ?TEID_A2U),
    %% [eNB -> MME] E-RAB SETUP RESPONSE
    SetupRspIn = e_rab_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    SetupRspExp = e_rab_setup_rsp_pdu(?ADDR_C2U, ?TEID_C2U),

    [?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, s1_setup_req_pdu())),
     ?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, s1_setup_rsp_pdu())),
     ?_assertEqual({forward, SetupReqExp}, s1ap_proxy:process_pdu(Pid, SetupReqIn)),
     ?_assertEqual({forward, SetupRspExp}, s1ap_proxy:process_pdu(Pid, SetupRspIn)),
     %% global counters
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_SETUP_REQ, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_SETUP_RSP, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_PROC, 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 2),
     %% per-eNB counters
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 2 + 2),
     ?_assertMetricENB(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_SETUP_REQ, 1),
     ?_assertMetricENB(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_SETUP_RSP, 1),
     ?_assertMetricENB(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2 + 2),
     ?_assertMetricENB(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_PROC, 2),
     ?_assertMetricENB(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 2),
     ?_assertMatch({ok, _}, s1ap_proxy:fetch_erab(Pid, {7, 9, 6})),
     ?_assertMatch([_], s1ap_proxy:fetch_erab_list(Pid))].


test_e_rab_setup_req_fail(#{handler := Pid}) ->
    %% pfcp_peer:session_establish_req/3 responds with a reject
    PDU = pfcp_mock:pdu_rsp_reject(session_establishment_response, ?SEID_Loc),
    pfcp_mock:mock_req(session_establish_req, PDU),
    %% eNB <- [S1GW <- MME] E-RAB SETUP REQUEST
    %% eNB -- [S1GW -> MME] E-RAB SETUP RESPONSE (failure)
    SetupReqIn = e_rab_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    SetupRspExp = e_rab_setup_rsp_fail_pdu(),

    [?_assertEqual({reply, SetupRspExp}, s1ap_proxy:process_pdu(Pid, SetupReqIn)),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_SETUP_REQ, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 0),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_PROC, 0),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_REPLY_ALL, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_REPLY_ERAB_SETUP_RSP, 1),
     ?_assertEqual([], s1ap_proxy:fetch_erab_list(Pid))].


test_e_rab_release_cmd(#{handler := Pid}) ->
    %% [eNB <- MME] E-RAB SETUP REQUEST
    SetupReq = e_rab_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    %% [eNB -> MME] E-RAB SETUP RESPONSE
    SetupRsp = e_rab_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    %% [eNB <- MME] E-RAB RELEASE COMMAND
    ReleaseCmd = e_rab_release_cmd_pdu(),
    %% [eNB -> MME] E-RAB RELEASE RESPONSE
    ReleaseRsp = e_rab_release_rsp_pdu(),

    [?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, SetupReq)),
     ?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, SetupRsp)),
     ?_assertMatch([_], s1ap_proxy:fetch_erab_list(Pid)),
     ?_assertEqual({forward, ReleaseCmd}, s1ap_proxy:process_pdu(Pid, ReleaseCmd)),
     ?_assertEqual({forward, ReleaseRsp}, s1ap_proxy:process_pdu(Pid, ReleaseRsp)),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_RELEASE_CMD, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_RELEASE_RSP, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_PROC, 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 2),
     ?_assertEqual([], s1ap_proxy:fetch_erab_list(Pid))].


test_e_rab_release_ind(#{handler := Pid}) ->
    %% [eNB <- MME] E-RAB SETUP REQUEST
    SetupReq = e_rab_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    %% [eNB -> MME] E-RAB SETUP RESPONSE
    SetupRsp = e_rab_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    %% [eNB -> MME] E-RAB RELEASE INDICATION
    ReleaseInd = e_rab_release_ind_pdu(),

    [?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, SetupReq)),
     ?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, SetupRsp)),
     ?_assertEqual({forward, ReleaseInd}, s1ap_proxy:process_pdu(Pid, ReleaseInd)),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 3),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_RELEASE_IND, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2 + 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 1),
     ?_assertEqual([], s1ap_proxy:fetch_erab_list(Pid))].


test_e_rab_modify_req_rsp(#{handler := Pid}) ->
    %% [eNB <- MME] E-RAB SETUP REQUEST
    SetupReq = e_rab_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    %% [eNB -> MME] E-RAB SETUP RESPONSE
    SetupRsp = e_rab_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    %% eNB <- [S1GW <- MME] E-RAB MODIFY REQUEST (new F-TEID)
    ModifyReqIn = e_rab_modify_req_pdu(?ADDR_U2CM, ?TEID_U2CM),
    %% [eNB <- S1GW] <- MME E-RAB MODIFY REQUEST
    %% for the eNB F-TEID remains unchanged
    ModifyReqExp = e_rab_modify_req_pdu(?ADDR_A2U, ?TEID_A2U),
    %% [eNB -> MME] E-RAB MODIFY RESPONSE
    ModifyRsp = e_rab_modify_rsp_pdu(),
    %% [eNB -> MME] E-RAB RELEASE INDICATION
    ReleaseInd = e_rab_release_ind_pdu(),

    [?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, SetupReq)),
     ?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, SetupRsp)),
     ?_assertEqual({forward, ModifyReqExp}, s1ap_proxy:process_pdu(Pid, ModifyReqIn)),
     ?_assertEqual({forward, ModifyRsp}, s1ap_proxy:process_pdu(Pid, ModifyRsp)),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_MODIFY_REQ, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_MODIFY_RSP, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_PROC, 2 + 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 1),
     ?_assertEqual({forward, ReleaseInd}, s1ap_proxy:process_pdu(Pid, ReleaseInd)),
     ?_assertEqual([], s1ap_proxy:fetch_erab_list(Pid))].


test_e_rab_modify_req_rsp_fail(#{handler := Pid}) ->
    %% [eNB <- MME] E-RAB SETUP REQUEST
    SetupReq = e_rab_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    %% [eNB -> MME] E-RAB SETUP RESPONSE
    SetupRsp = e_rab_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    %% [eNB <- MME] E-RAB MODIFY REQUEST (new F-TEID)
    ModifyReq = e_rab_modify_req_pdu(?ADDR_U2CM, ?TEID_U2CM),
    %% [eNB -> MME] E-RAB MODIFY RESPONSE (failure)
    ModifyRsp = e_rab_modify_rsp_fail_pdu(),
    %% [eNB -> MME] E-RAB RELEASE INDICATION
    ReleaseInd = e_rab_release_ind_pdu(),

    [?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, SetupReq)),
     ?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, SetupRsp)),
     ?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, ModifyReq)),
     ?_assertEqual({forward, ModifyRsp}, s1ap_proxy:process_pdu(Pid, ModifyRsp)),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_MODIFY_REQ, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_MODIFY_RSP, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_PROC, 2 + 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 1),
     ?_assertEqual({forward, ReleaseInd}, s1ap_proxy:process_pdu(Pid, ReleaseInd)),
     ?_assertEqual([], s1ap_proxy:fetch_erab_list(Pid))].


test_e_rab_modify_ind_cnf_modified(#{handler := Pid}) ->
    %% [eNB <- MME] E-RAB SETUP REQUEST
    SetupReq = e_rab_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    %% [eNB -> MME] E-RAB SETUP RESPONSE
    SetupRsp = e_rab_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    %% [eNB -> S1GW] -> MME E-RAB MODIFICATION INDICATION (new F-TEID)
    ModifyIndIn = e_rab_modify_ind_pdu(?ADDR_U2AM, ?TEID_U2AM),
    %% eNB -> [S1GW -> MME] E-RAB MODIFICATION INDICATION
    %% for the MME F-TEID remains unchanged
    ModifyIndExp = e_rab_modify_ind_pdu(?ADDR_C2U, ?TEID_C2U),
    %% [eNB <- MME] E-RAB MODIFICATION CONFIRMATION
    ModifyCnf = e_rab_modify_cnf_pdu(modified),
    %% [eNB -> MME] E-RAB RELEASE INDICATION
    ReleaseInd = e_rab_release_ind_pdu(),

    [?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, SetupReq)),
     ?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, SetupRsp)),
     ?_assertEqual({forward, ModifyIndExp}, s1ap_proxy:process_pdu(Pid, ModifyIndIn)),
     ?_assertEqual({forward, ModifyCnf}, s1ap_proxy:process_pdu(Pid, ModifyCnf)),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_MOD_IND, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_MOD_CNF, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_PROC, 2 + 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 1),
     ?_assertEqual({forward, ReleaseInd}, s1ap_proxy:process_pdu(Pid, ReleaseInd)),
     ?_assertEqual([], s1ap_proxy:fetch_erab_list(Pid))].


test_e_rab_modify_ind_cnf_not_modified(#{handler := Pid}) ->
    %% [eNB <- MME] E-RAB SETUP REQUEST
    SetupReq = e_rab_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    %% [eNB -> MME] E-RAB SETUP RESPONSE
    SetupRsp = e_rab_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    %% [eNB -> S1GW] -> MME E-RAB MODIFICATION INDICATION (new F-TEID)
    ModifyIndIn = e_rab_modify_ind_pdu(?ADDR_U2AM, ?TEID_U2AM),
    %% eNB -> [S1GW -> MME] E-RAB MODIFICATION INDICATION
    %% for the MME F-TEID remains unchanged
    ModifyIndExp = e_rab_modify_ind_pdu(?ADDR_C2U, ?TEID_C2U),
    %% [eNB <- MME] E-RAB MODIFICATION CONFIRMATION
    ModifyCnf = e_rab_modify_cnf_pdu(not_modified),
    %% [eNB -> MME] E-RAB RELEASE INDICATION
    ReleaseInd = e_rab_release_ind_pdu(),

    [?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, SetupReq)),
     ?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, SetupRsp)),
     ?_assertEqual({forward, ModifyIndExp}, s1ap_proxy:process_pdu(Pid, ModifyIndIn)),
     ?_assertEqual({forward, ModifyCnf}, s1ap_proxy:process_pdu(Pid, ModifyCnf)),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_MOD_IND, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_MOD_CNF, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_PROC, 2 + 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 1),
     ?_assertEqual({forward, ReleaseInd}, s1ap_proxy:process_pdu(Pid, ReleaseInd)),
     ?_assertEqual([], s1ap_proxy:fetch_erab_list(Pid))].


test_e_rab_modify_ind_cnf_release(#{handler := Pid}) ->
    %% [eNB <- MME] E-RAB SETUP REQUEST
    SetupReq = e_rab_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    %% [eNB -> MME] E-RAB SETUP RESPONSE
    SetupRsp = e_rab_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    %% [eNB -> S1GW] -> MME E-RAB MODIFICATION INDICATION (new F-TEID)
    ModifyIndIn = e_rab_modify_ind_pdu(?ADDR_U2AM, ?TEID_U2AM),
    %% eNB -> [S1GW -> MME] E-RAB MODIFICATION INDICATION
    %% for the MME F-TEID remains unchanged
    ModifyIndExp = e_rab_modify_ind_pdu(?ADDR_C2U, ?TEID_C2U),
    %% [eNB <- MME] E-RAB MODIFICATION CONFIRMATION
    ModifyCnf = e_rab_modify_cnf_pdu(release),

    [?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, SetupReq)),
     ?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, SetupRsp)),
     ?_assertEqual({forward, ModifyIndExp}, s1ap_proxy:process_pdu(Pid, ModifyIndIn)),
     ?_assertEqual({forward, ModifyCnf}, s1ap_proxy:process_pdu(Pid, ModifyCnf)),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_MOD_IND, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ERAB_MOD_CNF, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_PROC, 2 + 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 1),
     ?_assertEqual([], s1ap_proxy:fetch_erab_list(Pid))].


test_initial_context_setup(#{handler := Pid}) ->
    %% [eNB <- MME] INITIAL CONTEXT SETUP REQUEST
    InitCtxSetupReqIn = initial_context_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    InitCtxSetupReqExp = initial_context_setup_req_pdu(?ADDR_A2U, ?TEID_A2U),
    %% [eNB -> MME] INITIAL CONTEXT SETUP RESPONSE
    InitCtxSetupRspIn = initial_context_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    InitCtxSetupRspExp = initial_context_setup_rsp_pdu(?ADDR_C2U, ?TEID_C2U),

    [?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, s1_setup_req_pdu())),
     ?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, s1_setup_rsp_pdu())),
     ?_assertEqual({forward, InitCtxSetupReqExp}, s1ap_proxy:process_pdu(Pid, InitCtxSetupReqIn)),
     ?_assertEqual({forward, InitCtxSetupRspExp}, s1ap_proxy:process_pdu(Pid, InitCtxSetupRspIn)),
     %% global counters
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_INIT_CTX_REQ, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_INIT_CTX_RSP, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_PROC, 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 2),
     %% per-eNB counters
     ?_assertMetricENB(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 2 + 2),
     ?_assertMetricENB(?S1GW_CTR_S1AP_PROXY_IN_PKT_INIT_CTX_REQ, 1),
     ?_assertMetricENB(?S1GW_CTR_S1AP_PROXY_IN_PKT_INIT_CTX_RSP, 1),
     ?_assertMetricENB(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2 + 2),
     ?_assertMetricENB(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_PROC, 2),
     ?_assertMetricENB(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 2),
     ?_assertMatch([_], s1ap_proxy:fetch_erab_list(Pid))].


test_ue_ctx_release_req(#{handler := Pid}) ->
    %% [eNB <- MME] INITIAL CONTEXT SETUP REQUEST
    InitCtxSetupReq = initial_context_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    %% [eNB -> MME] INITIAL CONTEXT SETUP RESPONSE
    InitCtxSetupRsp = initial_context_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    %% [eNB -> MME] UE CONTEXT RELEASE REQUEST
    UeCtxReleaseReq = ue_ctx_release_req_pdu(),

    [?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, InitCtxSetupReq)),
     ?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, InitCtxSetupRsp)),
     ?_assertEqual({forward, UeCtxReleaseReq}, s1ap_proxy:process_pdu(Pid, UeCtxReleaseReq)),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 2 + 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_RELEASE_CTX_REQ, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2 + 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_PROC, 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 1),
     ?_assertMatch([], s1ap_proxy:fetch_erab_list(Pid))].


test_ue_ctx_release_cmd(#{handler := Pid}) ->
    %% [eNB <- MME] INITIAL CONTEXT SETUP REQUEST
    InitCtxSetupReq = initial_context_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    %% [eNB -> MME] INITIAL CONTEXT SETUP RESPONSE
    InitCtxSetupRsp = initial_context_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    %% [eNB <- MME] UE CONTEXT RELEASE COMMAND
    UeCtxReleaseCmd = ue_ctx_release_cmd_pdu(),
    %% [eNB -> MME] UE CONTEXT RELEASE COMPLETE
    UeCtxReleaseCompl = ue_ctx_release_compl_pdu(),

    [?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, InitCtxSetupReq)),
     ?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, InitCtxSetupRsp)),
     ?_assertEqual({forward, UeCtxReleaseCmd}, s1ap_proxy:process_pdu(Pid, UeCtxReleaseCmd)),
     ?_assertEqual({forward, UeCtxReleaseCompl}, s1ap_proxy:process_pdu(Pid, UeCtxReleaseCompl)),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_RELEASE_CTX_CMD, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_RELEASE_CTX_COMPL, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_ALL, 2 + 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_PROC, 2),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_OUT_PKT_FWD_UNMODIFIED, 2),
     ?_assertMatch([], s1ap_proxy:fetch_erab_list(Pid))].


test_drop_asn1_error(#{handler := Pid}) ->
    PDU = << 16#de, 16#ad, 16#be, 16#ef >>,
    [?_assertEqual({drop, PDU}, s1ap_proxy:process_pdu(Pid, PDU)),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_DROP_ALL, 1)].


test_drop_proc_error(#{handler := Pid}) ->
    %% [eNB <- MME] INITIAL CONTEXT SETUP REQUEST
    InitCtxSetupReq = initial_context_setup_req_pdu(?ADDR_U2C, ?TEID_U2C),
    %% [eNB -> MME] INITIAL CONTEXT SETUP RESPONSE
    InitCtxSetupRsp = initial_context_setup_rsp_pdu(?ADDR_U2A, ?TEID_U2A),
    %% [eNB <- MME] UE CONTEXT RELEASE COMMAND
    UeCtxReleaseCmd = ue_ctx_release_cmd_pdu(),

    [?_assertMatch({forward, _}, s1ap_proxy:process_pdu(Pid, InitCtxSetupReq)),
     %% eNB does not respond in time, so the MME releases the UE context
     ?_assertEqual({forward, UeCtxReleaseCmd}, s1ap_proxy:process_pdu(Pid, UeCtxReleaseCmd)),
     %% eNB is late to the party, perhaps because of increased link latency?
     ?_assertEqual({drop, InitCtxSetupRsp}, s1ap_proxy:process_pdu(Pid, InitCtxSetupRsp)),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_ALL, 3),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_PROC_ERROR, 1),
     ?_assertMetric(?S1GW_CTR_S1AP_PROXY_IN_PKT_DROP_ALL, 1)].


%% ------------------------------------------------------------------
%% S1AP PDU templates
%% ------------------------------------------------------------------

%% S1 SETUP REQUEST
s1_setup_req_pdu() ->
    << 16#00, 16#11, 16#00, 16#1a, 16#00, 16#00, 16#02, 16#00,
       16#3b, 16#40, 16#08, 16#00, 16#00, 16#f1, 16#10, 16#00,
       16#00, 16#00, 16#00, 16#00, 16#40, 16#00, 16#07, 16#00,
       16#0c, 16#0e, 16#40, 16#00, 16#f1, 16#10
    >>.


%% S1 SETUP RESPONSE
s1_setup_rsp_pdu() ->
    << 16#20, 16#11, 16#00, 16#29, 16#00, 16#00, 16#03, 16#00,
       16#3d, 16#40, 16#0e, 16#05, 16#80, 16#6f, 16#70, 16#65,
       16#6e, 16#35, 16#67, 16#73, 16#2d, 16#6d, 16#6d, 16#65,
       16#30, 16#00, 16#69, 16#00, 16#0b, 16#00, 16#00, 16#00,
       16#f1, 16#10, 16#00, 16#00, 16#00, 16#02, 16#00, 16#01,
       16#00, 16#57, 16#40, 16#01, 16#ff
    >>.


%% [eNB <- MME] E-RAB SETUP REQUEST
e_rab_setup_req_pdu(TLA, TEID) when is_binary(TLA),
                                    is_integer(TEID) ->
    << 16#00, 16#05, 16#00, 16#80, 16#9b, 16#00, 16#00, 16#03,
       16#00, 16#00, 16#00, 16#02, 16#00, 16#07, 16#00, 16#08,
       16#00, 16#02, 16#00, 16#09, 16#00, 16#10, 16#00, 16#80,
       16#87, 16#00, 16#00, 16#11, 16#00, 16#80, 16#81, 16#0c,
       16#00, 16#05, 16#04, 16#0f, 16#80,
       TLA/bytes, %% transportLayerAddress (IPv4)
       TEID:32/big, %% GTP-TEID
       16#72, 16#27, 16#c8,
       16#1a, 16#bc, 16#ec, 16#11, 16#62, 16#54, 16#c1, 16#01,
       16#05, 16#04, 16#03, 16#69, 16#6d, 16#73, 16#05, 16#01,
       16#c0, 16#a8, 16#65, 16#02, 16#5e, 16#02, 16#b3, 16#8c,
       16#58, 16#32, 16#27, 16#54, 16#80, 16#80, 16#21, 16#10,
       16#02, 16#00, 16#00, 16#10, 16#81, 16#06, 16#08, 16#08,
       16#08, 16#08, 16#83, 16#06, 16#08, 16#08, 16#04, 16#04,
       16#00, 16#0d, 16#04, 16#08, 16#08, 16#08, 16#08, 16#00,
       16#0d, 16#04, 16#08, 16#08, 16#04, 16#04, 16#00, 16#03,
       16#10, 16#20, 16#01, 16#48, 16#60, 16#48, 16#60, 16#00,
       16#00, 16#00, 16#00, 16#00, 16#00, 16#00, 16#00, 16#88,
       16#88, 16#00, 16#03, 16#10, 16#20, 16#01, 16#48, 16#60,
       16#48, 16#60, 16#00, 16#00, 16#00, 16#00, 16#00, 16#00,
       16#00, 16#00, 16#88, 16#44, 16#00, 16#0c, 16#04, 16#ac,
       16#16, 16#00, 16#15, 16#00, 16#10, 16#02, 16#05, 16#78
    >>.


%% [eNB -> MME] E-RAB SETUP RESPONSE
e_rab_setup_rsp_pdu(TLA, TEID) when is_binary(TLA),
                                    is_integer(TEID) ->
    << 16#20, 16#05, 16#00, 16#22, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#40, 16#02, 16#00, 16#07, 16#00, 16#08, 16#40,
       16#02, 16#00, 16#09, 16#00, 16#1c, 16#40, 16#0f, 16#00,
       16#00, 16#27, 16#40, 16#0a, 16#0c, 16#1f,
       TLA/bytes, %% transportLayerAddress (IPv4)
       TEID:32/big %% GTP-TEID
    >>.


%% [S1GW -> MME] E-RAB SETUP RESPONSE (failure)
%% Transport-cause: transport-resource-unavailable
e_rab_setup_rsp_fail_pdu() ->
    << 16#20, 16#05, 16#00, 16#1a, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#40, 16#02, 16#00, 16#07, 16#00, 16#08, 16#40,
       16#02, 16#00, 16#09, 16#00, 16#1d, 16#40, 16#07, 16#00,
       16#00, 16#23, 16#40, 16#02, 16#0c, 16#20
    >>.


%% [eNB <- MME] E-RAB RELEASE COMMAND
%% TODO: make E-RAB IDs configurable
e_rab_release_cmd_pdu() ->
    << 16#00, 16#07, 16#00, 16#1a, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#00, 16#02, 16#00, 16#07, 16#00, 16#08, 16#00,
       16#02, 16#00, 16#09, 16#00, 16#21, 16#40, 16#07, 16#00,
       16#00, 16#23, 16#40, 16#02, 16#0c, 16#40
    >>.


%% [eNB -> MME] E-RAB RELEASE RESPONSE
%% TODO: make E-RAB IDs configurable
e_rab_release_rsp_pdu() ->
    << 16#20, 16#07, 16#00, 16#19, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#00, 16#02, 16#00, 16#07, 16#00, 16#08, 16#00,
       16#02, 16#00, 16#09, 16#00, 16#45, 16#40, 16#06, 16#00,
       16#00, 16#0f, 16#40, 16#01, 16#0c
    >>.


%% [eNB -> MME] E-RAB RELEASE INDICATION
%% TODO: make E-RAB IDs configurable
e_rab_release_ind_pdu() ->
    << 16#00, 16#08, 16#00, 16#1a, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#00, 16#02, 16#00, 16#07, 16#00, 16#08, 16#00,
       16#02, 16#00, 16#09, 16#00, 16#6e, 16#40, 16#07, 16#00,
       16#00, 16#23, 16#40, 16#02, 16#0c, 16#40
    >>.


%% [eNB <- MME] E-RAB MODIFY REQUEST
e_rab_modify_req_pdu(TLA, TEID) when is_binary(TLA),
                                     is_integer(TEID) ->
    << 16#00, 16#06, 16#00, 16#2d, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#00, 16#02, 16#00, 16#07, 16#00, 16#08, 16#00,
       16#02, 16#00, 16#09, 16#00, 16#1e, 16#00, 16#1a, 16#00,
       16#00, 16#24, 16#00, 16#15, 16#4c, 16#00, 16#05, 16#04,
       16#00, 16#00, 16#00, 16#00, 16#b9, 16#00, 16#0a, 16#07,
       16#c0,
       TLA/bytes, %% transportLayerAddress (IPv4)
       TEID:32/big %% GTP-TEID
    >>.


%% [eNB -> MME] E-RAB MODIFY RESPONSE (success)
e_rab_modify_rsp_pdu() ->
    << 16#20, 16#06, 16#00, 16#19, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#40, 16#02, 16#00, 16#07, 16#00, 16#08, 16#40,
       16#02, 16#00, 16#09, 16#00, 16#1f, 16#40, 16#06, 16#00,
       16#00, 16#25, 16#40, 16#01, 16#0c
    >>.


%% [eNB -> MME] E-RAB MODIFY RESPONSE (failure)
e_rab_modify_rsp_fail_pdu() ->
    << 16#20, 16#06, 16#00, 16#1a, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#40, 16#02, 16#00, 16#07, 16#00, 16#08, 16#40,
       16#02, 16#00, 16#09, 16#00, 16#20, 16#40, 16#07, 16#00,
       16#00, 16#23, 16#40, 16#02, 16#0c, 16#40
    >>.


%% [eNB -> MME] E-RAB MODIFICATION INDICATION
e_rab_modify_ind_pdu(TLA, TEID) when is_binary(TLA),
                                     is_integer(TEID) ->
    << 16#00, 16#32, 16#00, 16#22, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#00, 16#02, 16#00, 16#07, 16#00, 16#08, 16#00,
       16#02, 16#00, 16#09, 16#00, 16#c7, 16#00, 16#0f, 16#00,
       16#00, 16#c8, 16#00, 16#0a, 16#0c, 16#1f,
       TLA/bytes, %% transportLayerAddress (IPv4)
       TEID:32/big %% GTP-TEID
    >>.


%% [eNB -> MME] E-RAB MODIFICATION CONFIRMATION
e_rab_modify_cnf_pdu(Res) ->
    IEs = case Res of
        modified ->
            << 16#00, 16#cb, 16#40, 16#06, 16#00,
               16#00, 16#cc, 16#40, 16#01, 16#0c >>;
        not_modified ->
            %% cause: transport: transport-resource-unavailable (0)
            << 16#00, 16#cd, 16#40, 16#07, 16#00,
               16#00, 16#23, 16#40, 16#02, 16#0c, 16#20 >>;
        release ->
            %% cause: transport: transport-resource-unavailable (0)
            << 16#00, 16#d2, 16#40, 16#07, 16#00,
               16#00, 16#23, 16#40, 16#02, 16#0c, 16#20 >>
    end,
    << 16#20, 16#32, 16#00, (15 + byte_size(IEs)),
       16#00, 16#00, 16#03, 16#00, 16#00, 16#40, 16#02, 16#00,
       16#07, 16#00, 16#08, 16#40, 16#02, 16#00, 16#09,
       IEs/bytes
    >>.


%% [eNB <- MME] INITIAL CONTEXT SETUP REQUEST
initial_context_setup_req_pdu(TLA, TEID) when is_binary(TLA),
                                              is_integer(TEID) ->
    << 16#00, 16#09, 16#00, 16#81, 16#10, 16#00, 16#00, 16#07,
       16#00, 16#00, 16#00, 16#02, 16#00, 16#01, 16#00, 16#08,
       16#00, 16#02, 16#00, 16#01, 16#00, 16#42, 16#00, 16#0a,
       16#18, 16#3b, 16#9a, 16#ca, 16#00, 16#60, 16#3b, 16#9a,
       16#ca, 16#00, 16#00, 16#18, 16#00, 16#80, 16#b5, 16#00,
       16#00, 16#34, 16#00, 16#80, 16#af, 16#45, 16#00, 16#09,
       16#20, 16#0f, 16#80,
       TLA/bytes, %% transportLayerAddress (IPv4)
       TEID:32/big, %% GTP-TEID
       16#80, 16#9f, 16#27, 16#bd, 16#de,
       16#61, 16#4e, 16#02, 16#07, 16#42, 16#02, 16#29, 16#06,
       16#40, 16#32, 16#f8, 16#10, 16#00, 16#01, 16#00, 16#6e,
       16#52, 16#05, 16#c1, 16#01, 16#09, 16#09, 16#08, 16#69,
       16#6e, 16#74, 16#65, 16#72, 16#6e, 16#65, 16#74, 16#05,
       16#01, 16#c0, 16#a8, 16#64, 16#02, 16#5e, 16#06, 16#fe,
       16#fe, 16#fa, 16#fa, 16#02, 16#02, 16#58, 16#32, 16#27,
       16#4d, 16#80, 16#80, 16#21, 16#10, 16#02, 16#00, 16#00,
       16#10, 16#81, 16#06, 16#08, 16#08, 16#08, 16#08, 16#83,
       16#06, 16#08, 16#08, 16#04, 16#04, 16#00, 16#0d, 16#04,
       16#08, 16#08, 16#08, 16#08, 16#00, 16#0d, 16#04, 16#08,
       16#08, 16#04, 16#04, 16#00, 16#03, 16#10, 16#20, 16#01,
       16#48, 16#60, 16#48, 16#60, 16#00, 16#00, 16#00, 16#00,
       16#00, 16#00, 16#00, 16#00, 16#88, 16#88, 16#00, 16#03,
       16#10, 16#20, 16#01, 16#48, 16#60, 16#48, 16#60, 16#00,
       16#00, 16#00, 16#00, 16#00, 16#00, 16#00, 16#00, 16#88,
       16#44, 16#00, 16#10, 16#02, 16#05, 16#aa, 16#50, 16#0b,
       16#f6, 16#32, 16#f8, 16#10, 16#00, 16#02, 16#01, 16#c0,
       16#00, 16#02, 16#86, 16#13, 16#32, 16#f8, 16#10, 16#00,
       16#01, 16#23, 16#05, 16#f4, 16#04, 16#94, 16#25, 16#f8,
       16#64, 16#02, 16#01, 16#08, 16#00, 16#6b, 16#00, 16#05,
       16#1c, 16#00, 16#0e, 16#00, 16#00, 16#00, 16#49, 16#00,
       16#20, 16#57, 16#5d, 16#d3, 16#66, 16#69, 16#73, 16#4c,
       16#8c, 16#48, 16#42, 16#76, 16#5f, 16#fe, 16#f9, 16#bd,
       16#d5, 16#89, 16#08, 16#ae, 16#f6, 16#f4, 16#88, 16#62,
       16#22, 16#d7, 16#80, 16#fb, 16#36, 16#33, 16#d9, 16#9e,
       16#9f, 16#00, 16#c0, 16#40, 16#08, 16#35, 16#87, 16#61,
       16#10, 16#03, 16#ff, 16#ff, 16#74
    >>.


%% [eNB -> MME] INITIAL CONTEXT SETUP RESPONSE
initial_context_setup_rsp_pdu(TLA, TEID) when is_binary(TLA),
                                              is_integer(TEID) ->
    << 16#20, 16#09, 16#00, 16#22, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#40, 16#02, 16#00, 16#01, 16#00, 16#08, 16#40,
       16#02, 16#00, 16#01, 16#00, 16#33, 16#40, 16#0f, 16#00,
       16#00, 16#32, 16#40, 16#0a, 16#0a, 16#1f,
       TLA/bytes, %% transportLayerAddress (IPv4)
       TEID:32/big %% GTP-TEID
    >>.


%% [eNB -> MME] UE CONTEXT RELEASE REQUEST
ue_ctx_release_req_pdu() ->
    << 16#00, 16#12, 16#40, 16#14, 16#00, 16#00, 16#03, 16#00,
       16#00, 16#00, 16#02, 16#00, 16#01, 16#00, 16#08, 16#00,
       16#02, 16#00, 16#01, 16#00, 16#02, 16#40, 16#01, 16#20
    >>.


%% [eNB <- MME] UE CONTEXT RELEASE COMMAND
ue_ctx_release_cmd_pdu() ->
    << 16#00, 16#17, 16#00, 16#10, 16#00, 16#00, 16#02, 16#00,
       16#63, 16#00, 16#04, 16#00, 16#01, 16#00, 16#01, 16#00,
       16#02, 16#40, 16#01, 16#20
    >>.


%% [eNB -> MME] UE CONTEXT RELEASE COMPLETE
ue_ctx_release_compl_pdu() ->
    << 16#20, 16#17, 16#00, 16#0f, 16#00, 16#00, 16#02, 16#00,
       16#00, 16#40, 16#02, 16#00, 16#01, 16#00, 16#08, 16#40,
       16#02, 16#00, 16#01
    >>.


%% vim:set ts=4 sw=4 et:
