/*
 * Copyright (C) 2012 Tobias Brunner
 * Copyright (C) 2008 Martin Willi
 * HSR Hochschule fuer Technik Rapperswil
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

/**
 * @defgroup rng rng
 * @{ @ingroup crypto
 */

#ifndef RNG_H_
#define RNG_H_

typedef enum rng_quality_t rng_quality_t;
typedef struct rng_t rng_t;

#include <library.h>

/**
 * Quality of generated random bytes.
 */
enum rng_quality_t {
	/** weak randomness, usable for nonces, IVs */
	RNG_WEAK,
	/** stronger randomness, usable for session keys */
	RNG_STRONG,
	/** true random key material */
	RNG_TRUE,
};

/**
 * enum name for rng_quality_t.
 */
extern enum_name_t *rng_quality_names;

/**
 * Generic interface for random number generators.
 */
struct rng_t {

	/**
	 * Generates random bytes and writes them in the buffer.
	 *
	 * @param len		number of bytes to get
	 * @param buffer	pointer where the generated bytes will be written
	 * @return			TRUE if bytes successfully written
	 */
	bool (*get_bytes)(rng_t *this, size_t len,
					  uint8_t *buffer) __attribute__((warn_unused_result));

	/**
	 * Generates random bytes and allocate space for them.
	 *
	 * @param len		number of bytes to get
	 * @param chunk		chunk which will hold generated bytes
	 * @return			TRUE if allocation succeeded
	 */
	bool (*allocate_bytes)(rng_t *this, size_t len,
						   chunk_t *chunk) __attribute__((warn_unused_result));

	/**
	 * Destroys a rng object.
	 */
	void (*destroy)(rng_t *this);
};

/**
 * Wrapper around rng_t.get_bytes() ensuring that either all bytes or at least
 * the first byte is not zero.
 *
 * @param rng			rng_t object
 * @param len			number of bytes to get
 * @param buffer		pointer where the generated bytes will be written
 * @param all			TRUE if all bytes have to be non-zero, FALSE for first
 * @return				TRUE if bytes successfully written
 */
bool rng_get_bytes_not_zero(rng_t *rng, size_t len, uint8_t *buffer,
							bool all) __attribute__((warn_unused_result));

/**
 * Wrapper around rng_t.allocate_bytes() ensuring that either all bytes or at
 * least the first byte is not zero.
 *
 * @param rng			rng_t object
 * @param len			number of bytes to get
 * @param chunk			chunk that stores the generated bytes (allocated)
 * @param all			TRUE if all bytes have to be non-zero, FALSE for first
 * @return				TRUE if bytes successfully written
 */
bool rng_allocate_bytes_not_zero(rng_t *rng, size_t len, chunk_t *chunk,
								 bool all) __attribute__((warn_unused_result));



#endif /** RNG_H_ @}*/
